/*
 * Copyright (C) 2014-2025 Mambo Solutions Ltd.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package io.mambo.sdk.service.behaviour.model;

import java.util.ArrayList;
import java.util.List;

import io.mambo.sdk.service.behaviour.data.BehaviourRequestData;
import io.mambo.sdk.service.behaviour.model.attributes.BehaviourAttrs;
import io.mambo.sdk.service.common.model.CustomFieldValueDto;
import io.mambo.sdk.service.common.model.PrizeDto;
import io.mambo.sdk.service.common.model.interfaces.HasCustomFields;
import io.mambo.sdk.service.common.model.interfaces.HasTranslation;
import io.mambo.sdk.service.common.model.tag.AbstractHasAwardByTagRequestData;
import io.mambo.sdk.utils.ListUtils;
import io.mambo.sdk.utils.StringUtils;
import lombok.EqualsAndHashCode;
import lombok.ToString;

/**
 * Behaviour data shared by {@link BehaviourDto} and {@link BehaviourRequestData}
 */
@ToString( callSuper = true )
@EqualsAndHashCode( callSuper = true )
public abstract class AbstractBehaviour extends AbstractHasAwardByTagRequestData
		implements HasTranslation, HasCustomFields
{
	private String name;
	private String verb;
	private String message;
	private String hint;
	private String startDate;
	private String endDate;
	private Integer coolOff;
	private Boolean active;
	private Boolean hideInWidgets;
	private Boolean jsTrackable;
	private BehaviourAttrs attrs;
	private PrizeDto prizes;
	private ActivityStreamDto activity;
	private LimitDto limit;
	private List<CustomFieldValueDto> customFields;
	private List<BehaviourTranslationDto> translations;

	/**
	 * The Behaviour's name
	 * @return
	 */
	public String getName() { return name; }
	public void setName( String name ) {
		initializedFields.add( "name" );
		this.name = name;
	}

	/**
	 * Indicates whether the behaviour is active or not.
	 * @return
	 */
	public Boolean getActive() { return active; }
	public void setActive( Boolean active ) {
		initializedFields.add( "active" );
		this.active = active;
	}

	/**
	 * The Behaviour's verb. This is used when creating a behaviour activity.
	 * @return
	 */
	public String getVerb() { return verb; }
	public void setVerb( String verb ) {
		initializedFields.add( "verb" );
		this.verb = verb;
	}

	/**
	 * The message associated with the behaviour.
	 * This will be displayed in the notifications.
	 * @return
	 */
	public String getMessage() { return message; }
	public void setMessage( String message ) {
		initializedFields.add( "message" );
		this.message = message;
	}

	/**
	 * The Behaviour's coolOff period. The time in seconds which must
	 * elapse before the user can earn points for this behaviour again.
	 * @return
	 */
	public Integer getCoolOff() { return coolOff; }
	public void setCoolOff( Integer coolOff ) {
		initializedFields.add( "coolOff" );
		this.coolOff = coolOff;
	}

	/**
	 * The Behaviour's hint. This is displayed to the end user when you
	 * wish to make them aware of what behaviours you are rewarding.
	 * @return
	 */
	public String getHint() { return hint; }
	public void setHint( String hint ) {
		initializedFields.add( "hint" );
		this.hint = hint;
	}

	/**
	 * Whether the Behaviour's hint should be shown or not.
	 * @return
	 */
	public Boolean getHideInWidgets() { return hideInWidgets; }
	public void setHideInWidgets( Boolean hideInWidgets ) {
		initializedFields.add( "hideInWidgets" );
		this.hideInWidgets = hideInWidgets;
	}

	/**
	 * Whether the Behaviour can be tracked directly through the Events JavaScript API
	 * @return
	 */
	public Boolean getJsTrackable() { return jsTrackable; }
	public void setJsTrackable( Boolean jsTrackable ) {
		initializedFields.add( "jsTrackable" );
		this.jsTrackable = jsTrackable;
	}

	/**
	 * This represents the date from which this behaviour can be performed by users.
	 * If no date is specified, the behaviour can always be performed.
	 * This must be a timestamp in ISO 8601 format with
	 * millisecond precision: YYYY-MM-DDTHH:MM:SS.MMMZ.
	 * @return
	 */
	public String getStartDate() { return startDate; }
	public void setStartDate( String startDate ) {
		initializedFields.add( "startDate" );
		this.startDate = startDate;
	}

	/**
	 * This represents the date from which this behaviour can no longer be performed by users
	 * If no date is specified, the behaviour can always be performed.
	 * This must be a timestamp in ISO 8601 format with
	 * millisecond precision: YYYY-MM-DDTHH:MM:SS.MMMZ.
	 * @return
	 */
	public String getEndDate() { return endDate; }
	public void setEndDate( String endDate ) {
		initializedFields.add( "endDate" );
		this.endDate = endDate;
	}

	/**
	 * The attributes of the behaviour. There are currently two types of
	 * attributes: SimpleAttrs and FlexibleAttrs.
	 * @return
	 */
	public BehaviourAttrs getAttrs() { return attrs; }
	public void setAttrs( BehaviourAttrs attrs ) {
		initializedFields.add( "attrs" );
		this.attrs = attrs;
	}

	/**
	 * The limit object is used to define whether there is a repetition
	 * limit on the behaviour. Limits can be either: 1) a hard limit,
	 * for example: 5 repetitions; 2) or a limit that expires, for example:
	 * 10 repetitions in a day
	 */
	public LimitDto getLimit() { return limit; }
	public void setLimit( LimitDto limit ) {
		initializedFields.add( "limit" );
		this.limit = limit;
	}

	/**
	 * The prizes object contains the prizes that a user will earn
	 * when performing this behaviour.
	 * @return
	 */
	public PrizeDto getPrizes() { return prizes; }
	public void setPrizes( PrizeDto prizes ) {
		initializedFields.add( "prizes" );
		this.prizes = prizes;
	}

	/**
	 * The activity object is used to define the text to be used
	 * in the activity stream.
	 * @return
	 */
	public ActivityStreamDto getActivity() { return activity; }
	public void setActivity( ActivityStreamDto activity ) {
		initializedFields.add( "activity" );
		this.activity = activity;
	}

	/**
	 * Custom fields defined for the behaviour. These can contain additional
	 * data or any kind of information you would like to store which isn't a
	 * standard field of the behaviour.
	 */
	@Override
	public List<CustomFieldValueDto> getCustomFields() { return customFields; }
	@Override
	public void setCustomFields( List<CustomFieldValueDto> customFields ) {
		initializedFields.add( "customFields" );
		this.customFields = customFields;
	}

	/**
	 * This contains the list of the translations which must be used with the object.
	 * Setting a property directly on the object (such as using setName()) will add
	 * the property to the list of translations using the default language of the server.
	 * If you set a property on the object directly (such as setName()) and then set
	 * the same property using setTranslations(), then the setTranslations() property
	 * will take priority.
	 */
	@Override
	public List<BehaviourTranslationDto> getTranslations() { return translations; }
	public void setTranslations( List<BehaviourTranslationDto> translations ) {
		initializedFields.add( "translations" );
		this.translations = translations;
	}
	public void addTranslation( BehaviourTranslationDto translation ) {
		if( translations == null ) {
			initializedFields.add( "translations" );
			translations = new ArrayList<>();
		}
		translations.add( translation );
	}

	/**
	 * Returns the translation for the languageCode supplied. If no translation
	 * can be found then null will be returned.
	 * @param languageCode
	 * @return
	 */
	public BehaviourTranslationDto getTranslation( String languageCode )
	{
		if( ListUtils.isEmpty( translations ) ) {
			return null;
		}

		for( BehaviourTranslationDto translation : translations ) {
			if( StringUtils.equals( languageCode, translation.getLanguageCode() ) ) {
				return translation;
			}
		}

		return null;
	}
}
