/*
 * Copyright (C) 2014-2025 Mambo Solutions Ltd.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package io.mambo.sdk.service.behaviour;

import java.io.File;
import java.util.List;

import io.mambo.sdk.http.HttpMethod;
import io.mambo.sdk.http.RequestOptions;
import io.mambo.sdk.http.ResponseType;
import io.mambo.sdk.http.api.ApiRequest;
import io.mambo.sdk.http.api.ApiRequestAdapter;
import io.mambo.sdk.service.behaviour.data.BehaviourRequestData;
import io.mambo.sdk.service.behaviour.model.BehaviourDto;
import io.mambo.sdk.service.behaviour.param.BehaviourCloneParams;
import io.mambo.sdk.service.behaviour.param.BehaviourCreateParams;
import io.mambo.sdk.service.behaviour.param.BehaviourCustomFieldParams;
import io.mambo.sdk.service.behaviour.param.BehaviourDeleteParams;
import io.mambo.sdk.service.behaviour.param.BehaviourGetFlexibleListParams;
import io.mambo.sdk.service.behaviour.param.BehaviourGetListParams;
import io.mambo.sdk.service.behaviour.param.BehaviourGetParams;
import io.mambo.sdk.service.behaviour.param.BehaviourGetSimpleListParams;
import io.mambo.sdk.service.behaviour.param.BehaviourUpdateParams;
import io.mambo.sdk.service.behaviour.param.BehaviourUploadParams;
import io.mambo.sdk.service.common.AbstractService;
import io.mambo.sdk.service.common.data.CustomFieldValueRequestData;
import io.mambo.sdk.service.common.data.DeleteRequestData;
import io.mambo.sdk.service.common.data.FileRequestData;
import io.mambo.sdk.service.common.model.CustomFieldValueDto;
import io.mambo.sdk.service.common.model.response.Status;

/**
 * The BehavioursService class handles all Behaviour related requests to the
 * Mambo API.
 */
public class BehavioursService extends AbstractService
{
    private static final String BEHAVIOURS_URI = "/v1/behaviours";
    private static final String BEHAVIOURS_ID_URI = BEHAVIOURS_URI + "/{behaviourId}";
    private static final String BEHAVIOURS_IMAGE_URI = BEHAVIOURS_ID_URI + "/image";
    private static final String BEHAVIOURS_CLONE_URI = BEHAVIOURS_ID_URI + "/clone";
    private static final String BEHAVIOURS_CUSTOM_URI = BEHAVIOURS_ID_URI + "/custom_fields";
    private static final String BEHAVIOURS_CUSTOM_IMAGE_URI = BEHAVIOURS_ID_URI + "/custom_fields/image";

    private static final String BEHAVIOURS_SITE_URI = "/v1/{siteUrl}/behaviours";
    private static final String SIMPLE_BEHAVIOURS_SITE_URI = BEHAVIOURS_SITE_URI + "/simple";
    private static final String FLEX_BEHAVIOURS_SITE_URI = BEHAVIOURS_SITE_URI + "/flexible";

    public BehavioursService( ApiRequestAdapter apiClient ) {
        super( apiClient );
    }


    /**
     * This method is used to create a new behaviour.
     *
     * @see BehaviourRequestData
     *
     * @param siteUrl
     *            The site to which the behaviour belongs to
     * @param data
     *            The behaviour request data
     * @return
     */
    public BehaviourDto create( String siteUrl, BehaviourRequestData data )
    {
        return create( BehaviourCreateParams.builder()
        	.siteUrl( siteUrl )
        	.build(), data );
    }


    /**
     * This method is used to create a new behaviour.
     *
     * @see BehaviourRequestData
     *
     * @param siteUrl
     *            The site to which the behaviour belongs to
     * @param data
     *            The behaviour request data
     * @param requestOptions
     *            The options to be used with this request
     * @return
     */
    public BehaviourDto create(
    	String siteUrl,
    	BehaviourRequestData data,
    	RequestOptions requestOptions )
    {
        return create( BehaviourCreateParams.builder()
        	.siteUrl( siteUrl )
        	.build(), data, requestOptions );
    }


    /**
     * This method is used to create a new behaviour.
     *
     * @param params
     *            The parameters required to create the behaviour
     * @param data
     *            The behaviour request data
     * @return
     */
    public BehaviourDto create( BehaviourCreateParams params, BehaviourRequestData data )
    {
        return create( params, data, RequestOptions.create() );
    }


    /**
     * This method is used to create a new behaviour.
     *
     * @param params
     *            The parameters required to create the behaviour
     * @param data
     *            The behaviour request data
     * @param requestOptions
     *            The options to be used with this request
     * @return
     */
    public BehaviourDto create(
    	BehaviourCreateParams params,
    	BehaviourRequestData data,
    	RequestOptions requestOptions )
    {
        return apiClient().request( ApiRequest.builder()
            .apiPath( BEHAVIOURS_SITE_URI )
            .responseClass( BehaviourDto.class )
            .method( HttpMethod.POST )
            .requestData( data )
            .options( requestOptions )
            .params( params )
            .build() );
    }


    /**
     * Update an existing behaviour by ID.
     *
     * @param behaviourId
     *            The ID of the behaviour to update
     * @param data
     *            The data with which to update the specified behaviour object
     * @return
     */
    public BehaviourDto update( String behaviourId, BehaviourRequestData data )
    {
        return update( BehaviourUpdateParams.builder()
        	.behaviourId( behaviourId )
        	.build(), data, RequestOptions.create() );
    }


    /**
     * Update an existing behaviour by ID.
     *
     * @param params
     *            The parameters required to update the behaviour
     * @param data
     *            The data with which to update the specified behaviour object
     * @return
     */
    public BehaviourDto update( BehaviourUpdateParams params, BehaviourRequestData data )
    {
        return update( params, data, RequestOptions.create() );
    }


    /**
     * Update an existing behaviour by ID.
     *
     * @param params
     *            The parameters required to update the behaviour
     * @param data
     *            The data with which to update the specified behaviour object
     * @param requestOptions
     *            The options to be used with this request
     * @return
     */
    public BehaviourDto update(
    	BehaviourUpdateParams params,
    	BehaviourRequestData data,
    	RequestOptions requestOptions )
    {
        return apiClient().request( ApiRequest.builder()
            .apiPath( BEHAVIOURS_ID_URI )
            .responseClass( BehaviourDto.class )
            .method( HttpMethod.PUT )
            .requestData( data )
            .options( requestOptions )
            .params( params )
            .build() );
    }


    /**
     * Upload an image for the behaviour
     *
     * @param behaviourId
     *            The ID of the behaviour for which to upload the image
     * @param image
     *            The image to upload for the behaviour
     * @return
     */
    public BehaviourDto uploadImage( String behaviourId, File image )
    {
        return uploadImage( BehaviourUploadParams.builder()
        	.behaviourId( behaviourId )
        	.build(), image, RequestOptions.create() );
    }


    /**
     * Upload an image for the behaviour
     *
     * @param params
     *            The parameters required to upload an image for the behaviour
     * @param image
     *            The image to upload for the behaviour
     * @return
     */
    public BehaviourDto uploadImage( BehaviourUploadParams params, File image )
    {
        return uploadImage( params, image, RequestOptions.create() );
    }


    /**
     * Upload an image for the behaviour
     *
     * @param params
     *            The parameters required to upload an image for the behaviour
     * @param image
     *            The image to upload for the behaviour
     * @param requestOptions
     *            The options to be used with this request
     * @return
     */
    public BehaviourDto uploadImage(
    	BehaviourUploadParams params,
    	File image,
    	RequestOptions requestOptions )
    {
        return apiClient().request( ApiRequest.builder()
            .apiPath( BEHAVIOURS_IMAGE_URI )
            .responseClass( BehaviourDto.class )
            .method( HttpMethod.POST )
            .requestData( new FileRequestData( image ) )
            .options( requestOptions )
            .params( params )
            .build() );
    }



    /**
     * Clone a behaviour
     *
     * @param behaviourId
     *            The ID of the behaviour to clone
     * @return
     */
    public BehaviourDto clone( String behaviourId )
    {
        return clone( BehaviourCloneParams.builder()
            .behaviourId( behaviourId )
            .build(), RequestOptions.create() );
    }


    /**
     * Clone a behaviour
     *
     * @param params
     *            The parameters required to clone the behaviour
     * @return
     */
    public BehaviourDto clone( BehaviourCloneParams params )
    {
        return clone( params, RequestOptions.create() );
    }


    /**
     * Clone a behaviour
     *
     * @param params
     *            The parameters required to clone the behaviour
     * @param requestOptions
     *            The options to be used with this request
     * @return
     */
    public BehaviourDto clone( BehaviourCloneParams params, RequestOptions requestOptions )
    {
        return apiClient().request( ApiRequest.builder()
            .apiPath( BEHAVIOURS_CLONE_URI )
            .responseClass( BehaviourDto.class )
            .method( HttpMethod.POST )
            .options( requestOptions )
            .params( params )
            .build() );
    }


    /**
     * Delete a behaviour by ID.
     *
     * @param behaviourId
     *            The ID of the behaviour to delete
     * @return
     */
    public Status delete( String behaviourId )
    {
        return delete( BehaviourDeleteParams.builder()
            .behaviourId( behaviourId )
            .build(), RequestOptions.create() );
    }


    /**
     * Delete a behaviour by ID.
     *
     * @param params
     *            The parameters required to delete the behaviour
     * @return
     */
    public Status delete( BehaviourDeleteParams params )
    {
        return delete( params, RequestOptions.create() );
    }


    /**
     * Delete a behaviour by ID.
     *
     * @param params
     *            The parameters required to delete the behaviour
     * @param requestOptions
     *            The options to be used with this request
     * @return
     */
    public Status delete( BehaviourDeleteParams params, RequestOptions requestOptions )
    {
        return apiClient().request( ApiRequest.builder()
            .apiPath( BEHAVIOURS_ID_URI )
            .responseClass( Status.class )
            .method( HttpMethod.DELETE )
            .options( requestOptions )
            .params( params )
            .build() );
    }


    /**
     * Get a behaviour by ID
     *
     * @param behaviourId
     *            The ID of the behaviour to retrieve
     * @return
     */
    public BehaviourDto get( String behaviourId )
    {
        return get( BehaviourGetParams.builder()
            .behaviourId( behaviourId )
            .build(), RequestOptions.create() );
    }


    /**
     * Get a behaviour by ID
     *
     * @param params
     *            The parameters required to retrieve the behaviour
     * @return
     */
    public BehaviourDto get( BehaviourGetParams params )
    {
        return get( params, RequestOptions.create() );
    }


    /**
     * Get a behaviour by ID
     *
     * @param params
     *            The parameters required to retrieve the behaviour
     * @param requestOptions
     *            The options to be used with this request
     * @return
     */
    public BehaviourDto get( BehaviourGetParams params, RequestOptions requestOptions )
    {
        return apiClient().request( ApiRequest.builder()
            .apiPath( BEHAVIOURS_ID_URI )
            .responseClass( BehaviourDto.class )
            .method( HttpMethod.GET )
            .options( requestOptions )
            .params( params )
            .build() );
    }


    /**
     * Delete a list of behaviours by ID
     *
     * @param data
     *            The {@link DeleteRequestData}
     * @return
     */
    public Status deleteBehaviours( DeleteRequestData data )
    {
        return deleteBehaviours( data, RequestOptions.create() );
    }


    /**
     * Delete a list of behaviours by ID
     *
     * @param data
     *            The {@link DeleteRequestData}
     * @param requestOptions
     *            The options to be used with this request
     * @return
     */
    public Status deleteBehaviours( DeleteRequestData data, RequestOptions requestOptions )
    {
        return apiClient().request( ApiRequest.builder()
            .apiPath( BEHAVIOURS_URI )
            .responseClass( Status.class )
            .method( HttpMethod.DELETE )
            .requestData( data )
            .options( requestOptions )
            .build() );
    }


    /**
     * Get a list of behaviours for the specified site.
     *
     * @param siteUrl
     *            The site from which to retrieve the behaviours
     * @return
     */
    public List<BehaviourDto> getBehaviours( String siteUrl )
    {
        return getBehaviours( BehaviourGetListParams.builder()
        	.siteUrl( siteUrl )
        	.build(), RequestOptions.create() );
    }


    /**
     * Get a list of behaviours for the specified site.
     *
     * @param params
     *            The parameters required to retrieve a list of behaviours
     * @return
     */
    public List<BehaviourDto> getBehaviours( BehaviourGetListParams params )
    {
        return getBehaviours( params, RequestOptions.create() );
    }


    /**
     * Get a list of behaviours for the specified site.
     *
     * @param params
     *            The parameters required to retrieve a list of behaviours
     * @param requestOptions
     *            The options to be used with this request
     * @return
     */
    public List<BehaviourDto> getBehaviours( BehaviourGetListParams params, RequestOptions requestOptions )
    {
        return apiClient().request( ApiRequest.builder()
            .apiPath( BEHAVIOURS_SITE_URI )
            .responseClass( BehaviourDto.class )
            .responseType( ResponseType.LIST )
            .method( HttpMethod.GET )
            .options( requestOptions )
            .params( params )
            .build() );
    }


    /**
     * Get a list of simple behaviours for the specified site.
     *
     * @param siteUrl
     *            The site from which to retrieve the behaviours
     * @return
     */
    public List<BehaviourDto> getSimpleBehaviours( String siteUrl )
    {
        return getSimpleBehaviours( BehaviourGetSimpleListParams.builder()
        	.siteUrl( siteUrl )
        	.build(), RequestOptions.create() );
    }



    /**
     * Get a list of simple behaviours for the specified site
     *
     * @param params
     *            The parameters required to retrieve a list of simple behaviours
     * @return
     */
    public List<BehaviourDto> getSimpleBehaviours( BehaviourGetSimpleListParams params )
    {
        return getSimpleBehaviours( params, RequestOptions.create() );
    }


    /**
     * Get a list of simple behaviours for the specified site
     *
     * @param params
     *            The parameters required to retrieve a list of simple behaviours
     * @param requestOptions
     *            The options to be used with this request
     * @return
     */
    public List<BehaviourDto> getSimpleBehaviours(
    	BehaviourGetSimpleListParams params,
    	RequestOptions requestOptions )
    {
        return apiClient().request( ApiRequest.builder()
            .apiPath( SIMPLE_BEHAVIOURS_SITE_URI )
            .responseClass( BehaviourDto.class )
            .responseType( ResponseType.LIST )
            .method( HttpMethod.GET )
            .options( requestOptions )
            .params( params )
            .build() );
    }


    /**
     * Get a list of flexible behaviours for the specified site.
     *
     * @param siteUrl
     *            The site from which to retrieve the behaviours
     * @return
     */
    public List<BehaviourDto> getFlexibleBehaviours( String siteUrl )
    {
        return getFlexibleBehaviours( BehaviourGetFlexibleListParams.builder()
        	.siteUrl( siteUrl )
        	.build(), RequestOptions.create() );
    }



    /**
     * Get a list of flexible behaviours for the specified site
     *
     * @param params
     *            The parameters required to retrieve a list of flexible behaviours
     * @return
     */
    public List<BehaviourDto> getFlexibleBehaviours( BehaviourGetFlexibleListParams params )
    {
        return getFlexibleBehaviours( params, RequestOptions.create() );
    }


    /**
     * Get a list of flexible behaviours for the specified site
     *
     * @param params
     *            The parameters required to retrieve a list of flexible behaviours
     * @param requestOptions
     *            The options to be used with this request
     * @return
     */
    public List<BehaviourDto> getFlexibleBehaviours(
    	BehaviourGetFlexibleListParams params,
    	RequestOptions requestOptions )
    {
        return apiClient().request( ApiRequest.builder()
            .apiPath( FLEX_BEHAVIOURS_SITE_URI )
            .responseClass( BehaviourDto.class )
            .responseType( ResponseType.LIST )
            .method( HttpMethod.GET )
            .options( requestOptions )
            .params( params )
            .build() );
    }


    /**
     * This method is used to add custom fields to an existing behaviour.
     *
     * @param behaviourId
     *            The behaviour to which custom fields should be added
     * @param data
     *            The custom field value request data
     * @return
     */
    public BehaviourDto addCustomFields( String behaviourId, CustomFieldValueRequestData data )
    {
        return addCustomFields( BehaviourCustomFieldParams.builder()
        	.behaviourId( behaviourId )
        	.build(), data, RequestOptions.create() );
    }


    /**
     * This method is used to add custom fields to an existing behaviour.
     *
     * @param params
     *            The parameters required to update the custom fields
     * @param data
     *            The custom field value request data
     * @return
     */
    public BehaviourDto addCustomFields( BehaviourCustomFieldParams params, CustomFieldValueRequestData data )
    {
        return addCustomFields( params, data, RequestOptions.create() );
    }


    /**
     * This method is used to add custom fields to an existing behaviour.
     *
     * @param params
     *            The parameters required to update the custom fields
     * @param data
     *            The custom field value request data
     * @param requestOptions
     *            The options to be used with this request
     * @return
     */
    public BehaviourDto addCustomFields(
    	BehaviourCustomFieldParams params,
    	CustomFieldValueRequestData data,
    	RequestOptions requestOptions )
    {
        return modCustomFields( HttpMethod.POST, params, data, requestOptions );
    }


    /**
     * This method is used to update custom fields of an existing behaviour.
     *
     * @param behaviourId
     *            The behaviour for which custom fields should be updated
     * @param data
     *            The custom field value request data
     * @return
     */
    public BehaviourDto updateCustomFields( String behaviourId, CustomFieldValueRequestData data )
    {
        return updateCustomFields( BehaviourCustomFieldParams.builder()
        	.behaviourId( behaviourId )
        	.build(), data, RequestOptions.create() );
    }


    /**
     * This method is used to update custom fields of an existing behaviour.
     *
     * @param params
     *            The parameters required to update the custom fields
     * @param data
     *            The custom field value request data
     * @return
     */
    public BehaviourDto updateCustomFields( BehaviourCustomFieldParams params, CustomFieldValueRequestData data )
    {
        return updateCustomFields( params, data, RequestOptions.create() );
    }


    /**
     * This method is used to update custom fields of an existing behaviour.
     *
     * @param params
     *            The parameters required to update the custom fields
     * @param data
     *            The custom field value request data
     * @param requestOptions
     *            The options to be used with this request
     * @return
     */
    public BehaviourDto updateCustomFields(
    	BehaviourCustomFieldParams params,
    	CustomFieldValueRequestData data,
    	RequestOptions requestOptions )
    {
        return modCustomFields( HttpMethod.PUT, params, data, requestOptions );
    }


    /**
     * This method is used to delete custom fields from an existing behaviour.
     *
     * @param behaviourId
     *            The behaviour from which custom fields should be delete
     * @param data
     *            The custom field value request data
     * @return
     */
    public BehaviourDto deleteCustomFields( String behaviourId, CustomFieldValueRequestData data )
    {
        return deleteCustomFields( BehaviourCustomFieldParams.builder()
        	.behaviourId( behaviourId )
        	.build(), data, RequestOptions.create() );
    }


    /**
     * This method is used to delete custom fields from an existing behaviour.
     *
     * @param params
     *            The parameters required to update the custom fields
     * @param data
     *            The custom field value request data
     * @return
     */
    public BehaviourDto deleteCustomFields( BehaviourCustomFieldParams params, CustomFieldValueRequestData data )
    {
        return deleteCustomFields( params, data, RequestOptions.create() );
    }


    /**
     * This method is used to delete custom fields from an existing behaviour.
     *
     * @param params
     *            The parameters required to update the custom fields
     * @param data
     *            The custom field value request data
     * @param requestOptions
     *            The options to be used with this request
     * @return
     */
    public BehaviourDto deleteCustomFields(
    	BehaviourCustomFieldParams params,
    	CustomFieldValueRequestData data,
    	RequestOptions requestOptions )
    {
        return modCustomFields( HttpMethod.DELETE, params, data, requestOptions );
    }


    /**
     * Generic method used to manipulate the custom fields
     *
     * @param method
     *            The HTTP method to use (POST, PUT, DELETE)
     * @param params
     *            The parameters required to update the custom fields
     * @param data
     *            The custom field value request data
     * @param requestOptions
     *            The options to be used with this request
     * @return
     */
    private BehaviourDto modCustomFields(
        HttpMethod method,
        BehaviourCustomFieldParams params,
        CustomFieldValueRequestData data,
        RequestOptions requestOptions )
    {
        return apiClient().request( ApiRequest.builder()
            .apiPath( BEHAVIOURS_CUSTOM_URI )
            .responseClass( BehaviourDto.class )
            .method( method )
            .params( params )
            .requestData( data )
            .options( requestOptions )
            .build() );
    }


    /**
     * This method is used to add a custom field with an image to an existing
     * behaviour.
     *
     * @param behaviourId
     *            The ID of the behaviour for which to upload the image
     * @param data
     *            The custom field value request data
     * @param image
     *            The image to upload for the behaviour
     * @return
     */
    public BehaviourDto addCustomField(
        String behaviourId,
        CustomFieldValueDto data,
        File image )
    {
        return addCustomField( BehaviourCustomFieldParams.builder()
        	.behaviourId( behaviourId )
        	.build(), data, image, RequestOptions.create() );
    }


    /**
     * This method is used to add a custom field with an image to an existing
     * behaviour.
     *
     * @param params
     *            The parameters required to update the custom fields
     * @param data
     *            The custom field value request data
     * @param image
     *            The image to upload for the behaviour
     * @return
     */
    public BehaviourDto addCustomField(
        BehaviourCustomFieldParams params,
        CustomFieldValueDto data,
        File image )
    {
        return addCustomField( params, data, image, RequestOptions.create() );
    }


    /**
     * This method is used to add a custom field with an image to an existing
     * behaviour.
     *
     * @param params
     *            The parameters required to update the custom fields
     * @param data
     *            The custom field value request data
     * @param image
     *            The image to upload for the behaviour
     * @param requestOptions
     *            The options to be used with this request
     * @return
     */
    public BehaviourDto addCustomField(
        BehaviourCustomFieldParams params,
        CustomFieldValueDto data,
        File image,
        RequestOptions requestOptions )
    {
        return apiClient().request( ApiRequest.builder()
            .apiPath( BEHAVIOURS_CUSTOM_IMAGE_URI )
            .responseClass( BehaviourDto.class )
            .method( HttpMethod.POST )
            .params( params )
            .requestData( new FileRequestData( image, data ) )
            .options( requestOptions )
            .build() );
    }


    /**
     * Return an empty {@link BehaviourRequestData} object
     *
     * @return
     */
    public BehaviourRequestData newBehaviourRequestData()
    {
        return new BehaviourRequestData();
    }
}