/*
 * Copyright (C) 2014-2025 Mambo Solutions Ltd.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package io.mambo.sdk.service.analytics.param;

import static io.mambo.sdk.utils.Preconditions.checkContainsKey;
import static io.mambo.sdk.utils.Preconditions.checkNotNull;

import io.mambo.sdk.service.common.param.AbstractServiceParamsBuilder;
import io.mambo.sdk.service.common.param.ServiceParams;
import io.mambo.sdk.utils.Multimap;

public class AnalyticsGetListParams implements ServiceParams
{
	private Multimap<String, String> params = new Multimap<>();

	private AnalyticsGetListParams( Builder builder ) {
		this.params = builder.params();
	}


	public static Builder builder()
	{
		return new Builder();
	}


	@Override
	public Multimap<String, String> asMap()
	{
		return params;
	}


	public static class Builder extends AbstractServiceParamsBuilder<Builder>
	{

		/**
		 * The site for which to retrieve the analytics
		 */
		public Builder siteUrl( String siteUrl )
		{
			checkNotNull( siteUrl, "siteUrl must not be null" );
			params.put( "siteUrl", siteUrl );
			return this;
		}


		/**
		 * Specifies the page of results to retrieve
		 */
		public Builder page( int page )
		{
			params.put( "page", String.valueOf( page ) );
			return this;
		}


		/**
		 * Specifies the number of results to retrieve, up to a maximum of 100
		 */
		public Builder count( int count )
		{
			params.put( "count", String.valueOf( count ) );
			return this;
		}


		/**
		 * Specifies the type of analytics data to return
		 */
		public Builder dataType( String dataType )
		{
			params.put( "dataType", dataType );
			return this;
		}


		/**
		 * Specifies the analytics report type to return
		 */
		public Builder reportType( String reportType )
		{
			params.put( "reportType", reportType );
			return this;
		}


		/**
		 * Specifies the start date of the date range filter in ISO 8601 format
		 */
		public Builder startDate( String startDate )
		{
			params.put( "startDate", startDate );
			return this;
		}


		/**
		 * Specifies the end date of the date range filter in ISO 8601 format
		 */
		public Builder endDate( String endDate )
		{
			params.put( "endDate", endDate );
			return this;
		}


		/**
		 * Specifies the tag by which to filter the analytics
		 */
		public Builder tag( String tag )
		{
			params.put( "tag", tag );
			return this;
		}


		public AnalyticsGetListParams build()
		{
			checkContainsKey( params, "siteUrl", "siteUrl must not be null" );
			return new AnalyticsGetListParams( this );
		}


		private Multimap<String, String> params()
		{
			return params;
		}
	}
}
