/*
 * Copyright (C) 2014-2025 Mambo Solutions Ltd.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package io.mambo.sdk.service.analytics.model.attributes.datapoint;

import java.util.List;
import java.util.Map;

import com.fasterxml.jackson.annotation.JsonIgnore;

import io.mambo.sdk.service.common.model.interfaces.HasInitializedData;
import io.mambo.sdk.utils.ListUtils;
import io.mambo.sdk.utils.NumberConverter;

/**
 * Defines analytics attributes specific to a point data point
 */
public class PointDataPoint implements HasInitializedData
{
	private transient List<String> initializedFields = ListUtils.newArrayList();

	private Map<String, Number> earnedByPointId;
	private Map<String, Number> redeemedByPointId;
	private Map<String, Number> expiredByPointId;
	private Map<String, Number> balanceByPointId;


	/**
	 * The total number of points earned by point ID
	 * @return
	 */
	public Map<String,Number> getEarnedByPointId() { return earnedByPointId; }
	public void setEarnedByPointId( Map<String, Number> earnedByPointId ) {
		initializedFields.add( "earnedByPointId" );
		this.earnedByPointId = earnedByPointId;
	}

	@JsonIgnore
	public Map<String,NumberConverter> getEarnedByPointIdConverter() {
		return NumberConverter.toNumberConverter( earnedByPointId );
	}


	/**
	 * The total number of points spent by point ID
	 * @return
	 */
	public Map<String,Number> getRedeemedByPointId() { return redeemedByPointId; }
	public void setRedeemedByPointId( Map<String, Number> redeemedByPointId ) {
		initializedFields.add( "redeemedByPointId" );
		this.redeemedByPointId = redeemedByPointId;
	}

	@JsonIgnore
	public Map<String,NumberConverter> getRedeemedByPointIdConverter() {
		return NumberConverter.toNumberConverter( redeemedByPointId );
	}


	/**
	 * The total number of points expired by point ID
	 * @return
	 */
	public Map<String,Number> getExpiredByPointId() { return expiredByPointId; }
	public void setExpiredByPointId( Map<String, Number> expiredByPointId ) {
		initializedFields.add( "expiredByPointId" );
		this.expiredByPointId = expiredByPointId;
	}

	@JsonIgnore
	public Map<String,NumberConverter> getExpiredByPointIdConverter() {
		return NumberConverter.toNumberConverter( expiredByPointId );
	}


	/**
	 * The total number of points balance by point ID
	 * @return
	 */
	public Map<String,Number> getBalanceByPointId() { return balanceByPointId; }
	public void setBalanceByPointId( Map<String, Number> balanceByPointId ) {
		initializedFields.add( "balanceByPointId" );
		this.balanceByPointId = balanceByPointId;
	}

	@JsonIgnore
	public Map<String,NumberConverter> getBalanceByPointIdConverter() {
		return NumberConverter.toNumberConverter( balanceByPointId );
	}


	@Override
	public List<String> getInitializedFields()
	{
		return initializedFields;
	}


	@Override
	public String toString()
	{
		StringBuilder builder = new StringBuilder();
		builder.append( "PointDataPoint [earnedByPointId=" );
		builder.append( earnedByPointId );
		builder.append( ", redeemedByPointId=" );
		builder.append( redeemedByPointId );
		builder.append( ", expiredByPointId=" );
		builder.append( expiredByPointId );
		builder.append( ", balanceByPointId=" );
		builder.append( balanceByPointId );
		builder.append( "][super: " );
		builder.append( super.toString() );
		builder.append( "]" );
		return builder.toString();
	}
}
