/*
 * Copyright (C) 2014-2025 Mambo Solutions Ltd.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package io.mambo.sdk.service.activity.param;

import static io.mambo.sdk.utils.Preconditions.checkContainsKey;
import static io.mambo.sdk.utils.Preconditions.checkNotNull;

import java.util.List;

import io.mambo.sdk.service.common.param.AbstractServiceParamsBuilder;
import io.mambo.sdk.service.common.param.ServiceParams;
import io.mambo.sdk.utils.Multimap;

public class ActivityGetListParams implements ServiceParams
{
	private Multimap<String, String> params = new Multimap<>();

	private ActivityGetListParams( Builder builder ) {
		this.params = builder.params();
	}


	public static Builder builder()
	{
		return new Builder();
	}


	@Override
	public Multimap<String, String> asMap()
	{
		return params;
	}


	public static class Builder extends AbstractServiceParamsBuilder<Builder>
	{

		/**
		 * The site for which to retrieve the list of objects
		 */
		public Builder siteUrl( String siteUrl )
		{
			checkNotNull( siteUrl, "siteUrl must not be null" );
			params.put( "siteUrl", siteUrl );
			return this;
		}


		/**
		 * Specifies the page of results to retrieve.
		 */
		public Builder page( int page )
		{
			params.put( "page", String.valueOf( page ) );
			return this;
		}


		/**
		 * Specifies the number of results to retrieve, up to a maximum of 100. Check
		 * the documentation to see which APIs support this option
		 */
		public Builder count( int count )
		{
			params.put( "count", String.valueOf( count ) );
			return this;
		}


		/**
		 * The list of tags to filter by (if any)
		 */
		public Builder tags( List<String> tags )
		{
			params.putAll( "tags", tags );
			return this;
		}


		/**
		 * Whether the tags should return items containing any one of the tags or
		 * whether the tags should return only items containing all of the tags.
		 * Possible values: hasAnyOf / hasAllOf Default: hasAnyOf
		 */
		public Builder tagsJoin( String tagsJoin )
		{
			params.put( "tagsJoin", tagsJoin );
			return this;
		}


		/**
		 * The tagUuid to use to filter the list by personalization tags
		 */
		public Builder tagUuid( String tagUuid )
		{
			params.put( "tagUuid", tagUuid );
			return this;
		}


		/**
		 * Specifies whether only activities which unlocked one or more behaviours
		 * should be returned.
		 */
		public Builder withBehavioursOnly( boolean withBehavioursOnly )
		{
			params.put( "withBehavioursOnly", String.valueOf( withBehavioursOnly ) );
			return this;
		}


		/**
		 * Specifies whether only activities which unlocked one or more rewards should
		 * be returned.
		 */
		public Builder withRewardsOnly( boolean withRewardsOnly )
		{
			params.put( "withRewardsOnly", String.valueOf( withRewardsOnly ) );
			return this;
		}


		/**
		 * Specifies whether only activities which unlocked one or more missions should
		 * be returned.
		 */
		public Builder withMissionsOnly( boolean withMissionsOnly )
		{
			params.put( "withMissionsOnly", String.valueOf( withMissionsOnly ) );
			return this;
		}


		/**
		 * Specifies whether only activities which unlocked one or more levels should be
		 * returned.
		 */
		public Builder withLevelsOnly( boolean withLevelsOnly )
		{
			params.put( "withLevelsOnly", String.valueOf( withLevelsOnly ) );
			return this;
		}


		/**
		 * Specifies whether only activities which unlocked one or more achievements
		 * should be returned.
		 */
		public Builder withAchievementsOnly( boolean withAchievementsOnly )
		{
			params.put( "withAchievementsOnly", String.valueOf( withAchievementsOnly ) );
			return this;
		}


		/**
		 * Specifies whether only activities which unlocked one or more gifts should be
		 * returned.
		 */
		public Builder withGiftsOnly( boolean withGiftsOnly )
		{
			params.put( "withGiftsOnly", String.valueOf( withGiftsOnly ) );
			return this;
		}


		/**
		 * Specifies whether only activities which changed one or more KPIs should be
		 * returned.
		 */
		public Builder withKPIsOnly( boolean withKPIsOnly )
		{
			params.put( "withKPIsOnly", String.valueOf( withKPIsOnly ) );
			return this;
		}


		/**
		 * Specifies whether only activities which were compensated should be returned
		 */
		public Builder withCompensatedOnly( boolean withCompensatedOnly )
		{
			params.put( "withCompensatedOnly", String.valueOf( withCompensatedOnly ) );
			return this;
		}


		/**
		 * Specifies whether only activities which have been rejected should be
		 * returned.
		 */
		public Builder withRejectedOnly( boolean withRejectedOnly )
		{
			params.put( "withRejectedOnly", String.valueOf( withRejectedOnly ) );
			return this;
		}


		/**
		 * Specifies whether the behaviour activities returned should include behaviours
		 * which have an exception status.
		 */
		public Builder withExceptions( boolean withExceptions )
		{
			params.put( "withExceptions", String.valueOf( withExceptions ) );
			return this;
		}


		/**
		 * Specifies whether the response should have nested objects populated.
		 */
		public Builder populated( boolean populated )
		{
			params.put( "populated", String.valueOf( populated ) );
			return this;
		}


		/**
		 * Sets the start date for filtering
		 */
		public Builder startDate( String startDate )
		{
			params.put( "startDate", startDate );
			return this;
		}


		/**
		 * Sets the end date for filtering
		 */
		public Builder endDate( String endDate )
		{
			params.put( "endDate", endDate );
			return this;
		}


		/**
		 * The list of point IDs to filter by (if any)
		 *
		 * @deprecated
		 */
		@Deprecated
		public Builder pointIds( List<String> pointIds )
		{
			params.putAll( "pointIds", pointIds );
			return this;
		}


		/**
		 * The list of reward IDs to filter by (if any)
		 */
		public Builder rewardIds( List<String> rewardIds )
		{
			params.putAll( "rewardIds", rewardIds );
			return this;
		}


		/**
		 * The list of behaviour IDs to filter by (if any)
		 */
		public Builder behaviourIds( List<String> behaviourIds )
		{
			params.putAll( "behaviourIds", behaviourIds );
			return this;
		}


		/**
		 * The list of KPI IDs to filter by (if any)
		 */
		public Builder kpiIds( List<String> kpiIds )
		{
			params.putAll( "kpiIds", kpiIds );
			return this;
		}


		public ActivityGetListParams build()
		{
			checkContainsKey( params, "siteUrl", "siteUrl must not be null" );
			return new ActivityGetListParams( this );
		}


		private Multimap<String, String> params()
		{
			return params;
		}
	}
}
