/*
 * Copyright (C) 2014-2025 Mambo Solutions Ltd.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package io.mambo.sdk.service.activity.model.attributes;

import java.util.ArrayList;
import java.util.List;

import io.mambo.sdk.service.common.model.PointStoreDto;
import io.mambo.sdk.service.common.model.interfaces.HasInitializedData;
import io.mambo.sdk.service.common.model.interfaces.HasPoints;
import io.mambo.sdk.utils.ListUtils;
import lombok.EqualsAndHashCode;
import lombok.ToString;

/**
 * Defines activity attributes specific to point activities.
 */
@ToString
@EqualsAndHashCode
public class ActivityPointAttrs implements ActivityAttrs, HasPoints, HasInitializedData
{
	private transient List<String> initializedFields = ListUtils.newArrayList( "type" );

	private String action;
	private List<PointStoreDto> points;


	/**
	 * The type of attribute
	 * @return
	 */
	@Override
	public String getType() { return "point"; }

	/**
	 * The action associated to the point activity. The actions currently
	 * available are:
	 * Increment: increments the user's total points and point balance
	 * Set: sets the user's total points and updates the point balance
	 * Redeem: decrements the user's points balance and increments the user's spent points
	 * Refund: decrements the user's spent points and increments the user's point balance
	 * @return
	 */
	public String getAction() { return action; }
	public void setAction( String action ) {
		initializedFields.add( "action" );
		this.action = action;
	}

	/**
	 * The points associated with this point activity.
	 * @return
	 */
	@Override
	public List<PointStoreDto> getPoints() { return points; }
	@Override
	public void setPoints( List<PointStoreDto> points ) {
		initializedFields.add( "points" );
		this.points = points;
	}
	@Override
	public void addPoint( PointStoreDto point ) {
		initializedFields.add( "points" );
		if( points == null ) {
			points = new ArrayList<>();
		}
		points.add( point );
	}


	@Override
	public List<String> getInitializedFields()
	{
		return initializedFields;
	}
}
