/*
 * Copyright (C) 2014-2025 Mambo Solutions Ltd.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package io.mambo.sdk.service.activity.model.attributes;

import java.util.ArrayList;
import java.util.List;

import io.mambo.sdk.service.activity.data.ActivityRequestData;
import io.mambo.sdk.service.common.model.PointStoreDto;
import io.mambo.sdk.service.common.model.interfaces.HasInitializedData;
import io.mambo.sdk.service.common.model.interfaces.HasPoints;
import io.mambo.sdk.service.user.model.UserDto;
import io.mambo.sdk.utils.ListUtils;
import lombok.EqualsAndHashCode;
import lombok.ToString;

/**
 * Defines activity attributes specific to gifted.
 */
@ToString
@EqualsAndHashCode
public class ActivityGiftedAttrs implements ActivityAttrs, HasPoints, HasInitializedData
{
	private transient List<String> initializedFields = ListUtils.newArrayList( "type" );

	private String targetUuid;
	private String targetUserId;
	private String rewardId;
	private List<PointStoreDto> points;
	private UserDto targetUser;


	/**
	 * The type of attribute
	 * @return
	 */
	@Override
	public String getType() { return "gifted"; }

	/**
	 * The Unique User ID of the user to whom the points are being gifted.
	 * This must be populated when sending a {@link ActivityRequestData}
	 * to the server but will be empty on response. Please see the
	 * targetUserId field for the response value.
	 * @return
	 */
	public String getTargetUuid() { return targetUuid; }
	public void setTargetUuid( String targetUuid ) {
		initializedFields.add( "targetUuid" );
		this.targetUuid = targetUuid;
	}

	/**
	 * The ID of the user to whom the points are being gifted. This is
	 * the response the server provides based on the targetUuid supplied.
	 * This should not be populated when submitting a gifted activity.
	 * @return
	 */
	public String getTargetUserId() { return targetUserId; }
	public void setTargetUserId( String targetUserId )
	{
		initializedFields.add( "targetUserId" );
		this.targetUserId = targetUserId;
	}

	/**
	 * The user to whom the points were gifted
	 * @return
	 */
	public UserDto getTargetUser() { return targetUser; }
	public void setTargetUser( UserDto targetUser ) {
		initializedFields.add( "targetUser" );
		this.targetUser = targetUser;
	}

	/**
	 * The points associated with this gifted activity.
	 * This must be populated when sending a {@link ActivityRequestData}
	 * to the server but will be empty on response.
	 * @return
	 */
	@Override
	public List<PointStoreDto> getPoints() { return points; }
	@Override
	public void setPoints( List<PointStoreDto> points ) {
		initializedFields.add( "points" );
		this.points = points;
	}
	@Override
	public void addPoint( PointStoreDto point ) {
		initializedFields.add( "points" );
		if( points == null ) {
			points = new ArrayList<>();
		}
		points.add( point );
	}

	/**
	 * The ID of the gift reward being gifted to the target user
	 *
	 * @return
	 */
	public String getRewardId() { return rewardId; }
	public void setRewardId( String rewardId ) {
		initializedFields.add( "rewardId" );
		this.rewardId = rewardId;
	}


	@Override
	public List<String> getInitializedFields()
	{
		return initializedFields;
	}
}
