/*
 * Copyright (C) 2014-2025 Mambo Solutions Ltd.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package io.mambo.sdk.service.activity.model.attributes;

import java.util.ArrayList;
import java.util.List;

import io.mambo.sdk.service.common.model.PointStoreDto;
import io.mambo.sdk.service.common.model.interfaces.HasInitializedData;
import io.mambo.sdk.service.common.model.interfaces.HasPoints;
import io.mambo.sdk.utils.ListUtils;
import lombok.EqualsAndHashCode;
import lombok.ToString;

/**
 * Defines activity attributes specific to bounty.
 */
@ToString
@EqualsAndHashCode
public class ActivityBountyAttrs implements ActivityAttrs, HasPoints, HasInitializedData
{
	private transient List<String> initializedFields = ListUtils.newArrayList( "type" );

	private String targetUserId;
	private String state;
	private String completedOn;
	private List<PointStoreDto> points;


	/**
	 * The type of attribute
	 * @return
	 */
	@Override
	public String getType() { return "bounty"; }

	/**
	 * The User ID of the user to whom the points are being bountied.
	 * This is not required when creating a bounty activity and will
	 * be populated in the server response if the bounty is awarded.
	 * @return
	 */
	public String getTargetUserId() { return targetUserId; }
	public void setTargetUserId( String targetUserId ) {
		initializedFields.add( "targetUserId" );
		this.targetUserId = targetUserId;
	}

	/**
	 * The state of the bounty activity.
	 * This is not required when creating a bounty activity and will
	 * be populated in the server response.
	 * @return
	 */
	public String getState() { return state; }
	public void setState( String state ) {
		initializedFields.add( "state" );
		this.state = state;
	}

	/**
	 * The date on which the bounty activity was completed.
	 * Completed means the activity was either cancelled or awarded.
	 * This will be a UTC time stamp in ISO 8601 format with
	 * millisecond precision: YYYY-MM-DDTHH:MM:SS.MMMZ
	 * This is not required when creating a bounty activity and will
	 * be populated in the server response if the bounty is completed
	 * (i.e. cancelled or awarded).
	 * @return
	 */
	public String getCompletedOn() { return completedOn; }
	public void setCompletedOn( String completedOn ) {
		initializedFields.add( "completedOn" );
		this.completedOn = completedOn;
	}

	/**
	 * The points associated with this bounty activity.
	 * @return
	 */
	@Override
	public List<PointStoreDto> getPoints() { return points; }
	@Override
	public void setPoints( List<PointStoreDto> points )
	{
		initializedFields.add( "points" );
		this.points = points;
	}
	@Override
	public void addPoint( PointStoreDto point )
	{
		initializedFields.add( "points" );
		if( points == null ) {
			points = new ArrayList<>();
		}
		points.add( point );
	}

	@Override
	public List<String> getInitializedFields()
	{
		return initializedFields;
	}
}
