/*
 * Copyright (C) 2014-2025 Mambo Solutions Ltd.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package io.mambo.sdk.service.activity.model.attributes;

import java.util.ArrayList;
import java.util.List;

import io.mambo.sdk.service.behaviour.model.metadata.MetadataDto;
import io.mambo.sdk.service.common.model.PointStoreDto;
import io.mambo.sdk.service.common.model.awarded.AwardedBehaviourDto;
import io.mambo.sdk.service.common.model.interfaces.HasInitializedData;
import io.mambo.sdk.utils.ListUtils;
import lombok.EqualsAndHashCode;
import lombok.ToString;

/**
 * Defines activity attributes specific to behaviour activities.
 */
@ToString
@EqualsAndHashCode
public class ActivityBehaviourAttrs implements ActivityAttrs, HasInitializedData
{
	protected transient List<String> initializedFields = ListUtils.newArrayList( "type" );

	private String verb;
	private String behaviourId;
	private List<MetadataDto> metadata = new ArrayList<>();
	private List<AwardedBehaviourDto> behaviours = new ArrayList<>();
	private List<PointStoreDto> adhocPoints;


	/**
	 * The type of attribute
	 * @return
	 */
	@Override
	public String getType() { return "behaviour"; }


	/**
	 * The behaviour performed by the user. Each behaviour has a corresponding verb
	 * which is the name of the behaviour in lower case. For example: The behaviour
	 * "Read an Article" would have an verb of "read an article".
	 * @return verb
	 */
	public String getVerb() { return verb; }
	public void setVerb( String verb ) {
		initializedFields.add( "verb" );
		this.verb = verb;
	}

	/**
	 * The ID of the behaviour performed by the user. Each behaviour has a corresponding
	 * ID which can be used to track a behaviour through the API. This is an alternative
	 * to using the behaviour's verb. If verb and behaviourId are populated, the verb will
	 * take precedence.
	 * @return behaviourId
	 */
	public String getBehaviourId() { return behaviourId; }
	public void setBehaviourId( String behaviourId ) {
		initializedFields.add( "behaviourId" );
		this.behaviourId = behaviourId;
	}

	/**
	 * The list of Metadata associated to this activity.
	 * @return
	 */
	public List<MetadataDto> getMetadata() { return metadata; }
	public void setMetadata( List<MetadataDto> metadata ) {
		initializedFields.add( "metadata" );
		this.metadata = metadata;
	}
	public void addMetadata( String name, String value ) {
		initializedFields.add( "metadata" );
		metadata.add( new MetadataDto( name, value ) );
	}


	/**
	 * Any ad hoc points associated with this behaviour activity.
	 * The ad hoc points will be added to the points of the simple
	 * behaviour if it's status is OK. Otherwise the ad hoc points
	 * are added to the first matching flexible behaviour with a
	 * status of OK. If there are no behaviours with a status of
	 * OK, then the points are not added.
	 * @return
	 */
	public List<PointStoreDto> getAdhocPoints() { return adhocPoints; }
	public void setAdhocPoints( List<PointStoreDto> adhocPoints ) {
		initializedFields.add( "adhocPoints" );
		this.adhocPoints = adhocPoints;
	}
	public void addAdhocPoint( PointStoreDto adhocPoint ) {
		initializedFields.add( "adhocPoints" );
		if( adhocPoints == null ) {
			adhocPoints = new ArrayList<>();
		}
		adhocPoints.add( adhocPoint );
	}


	/**
	 * The list of behaviours performed by the user in this activity.
	 * @return
	 */
	public List<AwardedBehaviourDto> getBehaviours() { return behaviours; }
	public void setBehaviours( List<AwardedBehaviourDto> behaviours ) {
		initializedFields.add( "behaviours" );
		this.behaviours = behaviours;
	}


	@Override
	public List<String> getInitializedFields()
	{
		return initializedFields;
	}
}
