/*
 * Copyright (C) 2014-2025 Mambo Solutions Ltd.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package io.mambo.sdk.service.activity.model;

import java.util.ArrayList;
import java.util.List;

import io.mambo.sdk.service.common.model.awarded.AwardedCouponDto;
import io.mambo.sdk.service.common.model.awarded.AwardedDto;
import io.mambo.sdk.service.common.model.awarded.AwardedKpiDto;
import io.mambo.sdk.service.common.model.awarded.AwardedPointDto;
import io.mambo.sdk.service.common.model.awarded.AwardedRewardDto;
import io.mambo.sdk.service.common.model.interfaces.AggregateDto;
import io.mambo.sdk.service.common.model.interfaces.HasId;
import io.mambo.sdk.service.common.model.tag.HasTags;
import io.mambo.sdk.service.tag.model.TagDto;
import io.mambo.sdk.service.user.model.UserDto;
import io.mambo.sdk.utils.ListUtils;
import io.mambo.sdk.utils.StringUtils;
import lombok.EqualsAndHashCode;
import lombok.ToString;


/**
 * This is returned when an activity is retrieved and provides information
 * regarding that activity e.g. the user who performed the activity, the
 * reward they unlocked (if any), etc.
 */
@ToString( callSuper = true )
@EqualsAndHashCode( callSuper = true )
public class ActivityDto extends AbstractActivity implements AggregateDto, HasTags, HasId
{
	private String id;
	private String userId;
	private String createdOn;
	private String modifiedOn;
	private String rejectedReason;
	private String state;
	private String systemInfo;
	private UserDto user;
	private UserTagsDto userTags;
	private List<TagDto> tags;
	private List<String> tagIds;
	private List<String> rewardIds = new ArrayList<>();
	private List<AwardedDto> awarded = new ArrayList<>();

	private List<AwardedKpiDto> kpis = new ArrayList<>();
	private List<AwardedRewardDto> rewards = new ArrayList<>();
	private List<AwardedPointDto> points = new ArrayList<>();
	private List<AwardedCouponDto> coupons = new ArrayList<>();


	/**
	 * This ID uniquely identifies this object
	 */
	@Override
	public String getId() { return id; }
	public void setId( String id ) { this.id = id; }


	/**
	 * The ID of the user who performed the activity.
	 * @return
	 */
	public String getUserId() { return userId; }
	public void setUserId( String userId ) { this.userId = userId; }


	/**
	 * If the activity was initiated by a trigger in the system
	 * then this will contain what caused this activity to be
	 * triggered. For example, updating a KPIs default targets
	 * may trigger a system activity.
	 * @return
	 */
	public String getSystemInfo() { return systemInfo; }
	public void setSystemInfo( String systemInfo ) { this.systemInfo = systemInfo; }

	/**
	 * The tags of the user who performed the activity
	 * @return
	 */
	public UserTagsDto getUserTags() { return userTags; }
	public void setUserTags( UserTagsDto userTags ) { this.userTags = userTags; }


	/**
	 * The list of awarded objects unlocked by the user by performing the activity.
	 * @return
	 */
	public List<AwardedDto> getAwarded() { return awarded; }
	public void setAwarded( List<AwardedDto> awarded ) { this.awarded = awarded; }


	/**
	 * The list of rewards unlocked by the user by performing the activity.
	 * @return
	 */
	public List<AwardedRewardDto> getRewards()
	{
		if( ListUtils.isEmpty( awarded ) )
		{
			return rewards;
		}

		if( ListUtils.isNotEmpty( rewards ) )
		{
			return rewards;
		}

		for( AwardedDto awd : awarded ) {
			if( StringUtils.equals( "reward", awd.getAwardedType() ) ) {
				rewards.add( (AwardedRewardDto) awd );
			}
		}
		return rewards;
	}


	/**
	 * The list of KPIs unlocked by the user by performing the activity.
	 * @return
	 */
	public List<AwardedKpiDto> getKpis()
	{
		if( ListUtils.isEmpty( awarded ) )
		{
			return kpis;
		}

		if( ListUtils.isNotEmpty( kpis ) )
		{
			return kpis;
		}

		for( AwardedDto awd : awarded ) {
			if( StringUtils.equals( "kpi", awd.getAwardedType() ) ) {
				kpis.add( (AwardedKpiDto) awd );
			}
		}
		return kpis;
	}


	/**
	 * The list of reward IDs unlocked by the user by performing the activity.
	 * @return
	 */
	public List<String> getRewardIds() { return rewardIds; }
	public void setRewardIds( List<String> rewardIds ) { this.rewardIds = rewardIds; }


	/**
	 * The time and date on which the activity was created.
	 *
	 * This will be a UTC timestamp in ISO 8601 format with
	 * millisecond precision: YYYY-MM-DDTHH:MM:SS.MMMZ
	 * @return
	 */
	public String getCreatedOn() { return  createdOn; }
	public void setCreatedOn( String createdOn ) { this.createdOn = createdOn; }

	/**
	 * The last date on which the activity was modified.
	 * This will be a UTC timestamp in ISO 8601 format with
	 * millisecond precision: YYYY-MM-DDTHH:MM:SS.MMMZ
	 * For example: 2013-01-20T20:43:24.094Z
	 * @return
	 */
	public String getModifiedOn() { return modifiedOn; }
	public void setModifiedOn( String modifiedOn ) { this.modifiedOn = modifiedOn; }

	/**
	 * If the activity is rejected, this will contain the reason
	 * why it was rejected.
	 * @return
	 */
	public String getRejectedReason() { return rejectedReason; }
	public void setRejectedReason( String rejectedReason ) { this.rejectedReason = rejectedReason; }

	/**
	 * The state of the activity. An activity can be in one
	 * of three states: approved, rejected or expired.
	 * @return
	 */
	public String getState() { return state; }
	public void setState( String state ) { this.state = state; }

	/**
	 * The list of points unlocked by the user who performed this activity
	 * @return
	 */
	public List<AwardedPointDto> getPoints() { return points; }
	public void setPoints( List<AwardedPointDto> points ) { this.points = points; }

	/**
	 * The list of coupons unlocked by the user who performed this activity
	 * @return
	 */
	public List<AwardedCouponDto> getCoupons() { return coupons; }
	public void setCoupons( List<AwardedCouponDto> coupons ) { this.coupons = coupons; }

	/**
	 * This should contain the list of the IDs of the tags which
	 * must be added to the object.
	 * @return
	 */
	@Override
	public List<String> getTagIds() { return tagIds; }
	@Override
	public void setTagIds( List<String> tagIds ) {
		this.tagIds = tagIds;
	}

	/**
	 * This contains the list of {@link TagDto} associated to this object.
	 * @return
	 */
	@Override
	public List<TagDto> getTags() { return tags; }
	@Override
	public void setTags( List<TagDto> tags ) {
		this.tags = tags;
	}


	/**
	 * The User who performed the activity
	 * @return
	 */
	public UserDto getUser() { return user; }
	public void setUser( UserDto user ) { this.user = user; }
}
