/*
 * Copyright (C) 2014-2025 Mambo Solutions Ltd.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package io.mambo.sdk.service.activity.model;

import java.util.ArrayList;
import java.util.List;

import io.mambo.sdk.service.activity.data.ActivityRequestData;
import io.mambo.sdk.service.activity.model.attributes.ActivityAttrs;
import io.mambo.sdk.service.common.data.RequestData;
import io.mambo.sdk.utils.StringUtils;
import lombok.EqualsAndHashCode;
import lombok.ToString;

/**
 * Tag data shared by {@link ActivityDto} and {@link ActivityRequestData}
 */
@ToString
@EqualsAndHashCode
public abstract class AbstractActivity implements RequestData
{
	protected transient List<String> initializedFields = new ArrayList<>();

	public enum ActivityType {
		BEHAVIOUR("behaviour"), COUPON("coupon"), POINT("point"),
		BOUNTY("bounty"), GIFTED("gifted"), KPI("kpi");
		private String value;
		private ActivityType( String value ) { this.value = value; }
		public String value() { return value; }
		public Boolean isValue( String other ) { return StringUtils.equals( value, other ); }
	}

	private String url;
	private String reason;
	private ContentDto content;
	private ContentDto target;
	private ActivityAttrs attrs;


	/**
	 * The URL of the page on which the activity took place. This should
	 * be a fully qualified URL. For example: http://www.acme.com/the/page.html
	 * @return url
	 */
	public String getUrl() { return url; }
	public void setUrl( String url ) {
		initializedFields.add( "url" );
		this.url = url;
	}

	/**
	 * The reason associated to the activity. This can be used for the
	 * source of the activity to provide a message / reason associated
	 * to the activity. This is used when displaying information to the
	 * users.
	 * @return
	 */
	public String getReason() { return reason; }
	public void setReason( String reason ) {
		initializedFields.add( "reason" );
		this.reason = reason;
	}


	/**
	 * This {@link ContentDto} represents the piece of content on which the
	 * behaviour took place, if any. For example, if the user shares a blog
	 * post, the content could represent the blog post. The content is then
	 * used when generating the activity streams to create rich messages for
	 * the users. Using the blog example we could display:
	 * Bob Dylan has shared [A Guide to Copy Writing]
	 * Where the [] represent a link to the blog post.
	 * @return verb
	 */
	public ContentDto getContent() { return content; }
	public void setContent( ContentDto content ) {
		initializedFields.add( "content" );
		this.content = content;
	}


	/**
	 * This {@link ContentDto} represents the target of the content. Using the
	 * Content field example, the user might share the blog to Facebook, the
	 * target object can be used to capture this:
	 * Bob Dylan has shared [A Guide to Copy Writing] to [Facebook]
	 * Where the [] represent a link to the blog post and Facebook.
	 * Another example could be uploading an image to a specific photo album,
	 * that might look something like this:
	 * Bob Dylan has uploaded [Relaxing on a beach] to [Epic Journey]
	 * @return verb
	 */
	public ContentDto getTarget() { return target; }
	public void setTarget( ContentDto target ) {
		initializedFields.add( "target" );
		this.target = target;
	}

	/**
	 * The attributes of the activity
	 * @return
	 */
	public ActivityAttrs getAttrs() { return attrs; }
	public void setAttrs( ActivityAttrs attrs ) {
		initializedFields.add( "attrs" );
		this.attrs = attrs;
	}

	/**
	 * True if the activity is a behaviour activity
	 * @return
	 */
	public Boolean isBehaviour() {
		if( attrs == null ) {
			return false;
		}
		return ActivityType.BEHAVIOUR.isValue( attrs.getType() );
	}

	/**
	 * True if the activity is a coupon activity
	 * @return
	 */
	public Boolean isCoupon() {
		if( attrs == null ) {
			return false;
		}
		return ActivityType.COUPON.isValue( attrs.getType() );
	}

	/**
	 * True if the activity is a point activity
	 * @return
	 */
	public Boolean isPoint() {
		if( attrs == null ) {
			return false;
		}
		return ActivityType.POINT.isValue( attrs.getType() );
	}

	/**
	 * True if the activity is a bounty activity
	 * @return
	 */
	public Boolean isBounty() {
		if( attrs == null ) {
			return false;
		}
		return ActivityType.BOUNTY.isValue( attrs.getType() );
	}

	/**
	 * True if the activity is a gifted activity
	 * @return
	 */
	public Boolean isGifted() {
		if( attrs == null ) {
			return false;
		}
		return ActivityType.GIFTED.isValue( attrs.getType() );
	}

	/**
	 * True if the activity is a KPI activity
	 * @return
	 */
	public Boolean isKPI() {
		if( attrs == null ) {
			return false;
		}
		return ActivityType.KPI.isValue( attrs.getType() );
	}


	@Override
	public List<String> getInitializedFields() {
		return initializedFields;
	}
}
