/*
 * Copyright (C) 2014-2025 Mambo Solutions Ltd.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package io.mambo.sdk.service.activity.data;

import java.util.ArrayList;
import java.util.List;

import io.mambo.sdk.service.common.data.RequestData;
import io.mambo.sdk.service.common.model.tag.HasTagsRequestData;
import io.mambo.sdk.service.tag.model.TagDto;
import io.mambo.sdk.utils.TagUtils;
import lombok.EqualsAndHashCode;
import lombok.ToString;

/**
 * This object captures the data required by the Activity API in
 * order to reject activities in bulk by criteria and site.
 */
@ToString
@EqualsAndHashCode
public class RejectActivityCriteriaRequestData implements RequestData, HasTagsRequestData
{
	protected transient List<String> initializedFields = new ArrayList<>();

	private String reason;
	private String startDate;
	private String endDate;
	private String verb;
	private String uuid;
	private String targetUserUuid;
	private String tagsJoin;
	private List<TagDto> tagDtos;
	private List<String> tagIds;
	private List<String> tagTags;
	private List<String> pointIds;
	private List<String> rewardIds;
	private List<String> kpiIds;
	private List<String> behaviourIds;


	/**
	 * The reason why the activity was rejected. This is used to give the user
	 * insight as to why the activity was rejected.
	 */
	public String getReason() { return reason; }
	public void setReason( String reason ) {
		initializedFields.add( "reason" );
		this.reason = reason;
	}

	/**
	 * This represents the start date from which the activities should be rejected
	 * This must be a UTC timestamp in ISO 8601 format with
	 * millisecond precision: YYYY-MM-DDTHH:MM:SS.MMMZ
	 * @return
	 */
	public String getStartDate() { return startDate; }
	public void setStartDate( String startDate ) {
		initializedFields.add( "startDate" );
		this.startDate = startDate;
	}

	/**
	 * This represents the last date until which the activities should be rejected
	 * This must be a UTC timestamp in ISO 8601 format with
	 * millisecond precision: YYYY-MM-DDTHH:MM:SS.MMMZ
	 * @return
	 */
	public String getEndDate() { return endDate; }
	public void setEndDate( String endDate ) {
		initializedFields.add( "endDate" );
		this.endDate = endDate;
	}


	/**
	 * Reject activities that have a matching verb
	 * @return
	 */
	public String getVerb() { return verb; }
	public void setVerb( String verb ) {
		initializedFields.add( "verb" );
		this.verb = verb;
	}


	/**
	 * Reject activities that have a matching UUID
	 * @return
	 */
	public String getUuid() { return uuid; }
	public void setUuid( String uuid ) {
		initializedFields.add( "uuid" );
		this.uuid = uuid;
	}


	/**
	 * Reject activities that have a matching target user UUID
	 * @return
	 */
	public String getTargetUserUuid() { return targetUserUuid; }
	public void setTargetUserUuid( String targetUserUuid ) {
		initializedFields.add( "targetUserUuid" );
		this.targetUserUuid = targetUserUuid;
	}


	/**
	 * Reject activities that have matching tags. Note that this will
	 * rely on the tagsJoin to determine whether we want all tags to
	 * be present or any 1 of the tags.
	 * @return
	 */
	@Override
	public List<String> getTagIds() { return tagIds; }
	@Override
	public void setTagIds( List<String> tagIds ) {
		initializedFields.add( "tagIds" );
		this.tagIds = tagIds;
	}


	/**
	 * Reject activities that have matching tags. Note that this will
	 * rely on the tagsJoin to determine whether we want all tags to
	 * be present or any 1 of the tags.
	 * @return
	 */
	@Override
	public List<TagDto> getTags() { return tagDtos; }
	@Override
	public void setTags( List<TagDto> tagDtos ) {
		// DO not call initializedFields here as we
		// don't want to send the TagDtos to the server
		setTagIds( TagUtils.extractTagIds( tagDtos ) );
		this.tagDtos = tagDtos;
	}


	/**
	 * Reject activities that have matching tags. Note that this will
	 * rely on the tagsJoin to determine whether we want all tags to
	 * be present or any 1 of the tags.
	 * @return
	 */
	@Override
	public List<String> getTagTags() { return tagTags; }
	@Override
	public void setTagTags( List<String> tagTags ) {
		initializedFields.add( "tagTags" );
		this.tagTags = tagTags;
	}


	/**
	 * Joins the tags provided either with hasAnyOf or hasAllOf
	 * @return
	 */
	public String getTagsJoin() { return tagsJoin; }
	public void setTagsJoin( String tagsJoin ) {
		initializedFields.add( "tagsJoin" );
		this.tagsJoin = tagsJoin;
	}


	/**
	 * Reject activities that have matching point IDs
	 * @return
	 */
	public List<String> getPointIds() { return pointIds; }
	public void setPointIds( List<String> pointIds ) {
		initializedFields.add( "pointIds" );
		this.pointIds = pointIds;
	}


	/**
	 * Reject activities that have matching reward IDs
	 * @return
	 */
	public List<String> getRewardIds() { return rewardIds; }
	public void setRewardIds( List<String> rewardIds ) {
		initializedFields.add( "rewardIds" );
		this.rewardIds = rewardIds;
	}


	/**
	 * Reject activities that have matching KPI IDs
	 * @return
	 */
	public List<String> getKpiIds() { return kpiIds; }
	public void setKpiIds( List<String> kpiIds ) {
		initializedFields.add( "kpiIds" );
		this.kpiIds = kpiIds;
	}


	/**
	 * Reject activities that have matching behaviour IDs
	 * @return
	 */
	public List<String> getBehaviourIds() { return behaviourIds; }
	public void setBehaviourIds( List<String> behaviourIds ) {
		initializedFields.add( "behaviourIds" );
		this.behaviourIds = behaviourIds;
	}


	@Override
	public List<String> getInitializedFields() {
		return initializedFields;
	}
}