/*
 * Copyright (C) 2014-2025 Mambo Solutions Ltd.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package io.mambo.sdk.service.activity.data;

import java.util.ArrayList;
import java.util.List;

import com.fasterxml.jackson.annotation.JsonTypeName;

import io.mambo.sdk.service.activity.model.attributes.ActivityAttrs;
import io.mambo.sdk.service.common.model.PointStoreDto;
import io.mambo.sdk.utils.Multimap;
import lombok.EqualsAndHashCode;
import lombok.ToString;

/**
 * Defines activity attributes specific to behaviour activities.
 */
@ToString
@EqualsAndHashCode
@JsonTypeName( "behaviour" )
public class ActivityBehaviourAttrsData implements ActivityAttrs
{
	private String verb;
	private String behaviourId;
	private Multimap<String,String> metadata = new Multimap<>();
	private List<PointStoreDto> adhocPoints;


	/**
	 * The type of attribute
	 * @return
	 */
	@Override
	public String getType() { return "behaviour"; }


	/**
	 * The behaviour performed by the user. Each behaviour has a corresponding verb
	 * which is the name of the behaviour in lower case. For example: The behaviour
	 * "Read an Article" would have an verb of "read an article".
	 * @return verb
	 */
	public String getVerb() { return verb; }
	public void setVerb( String verb ) { this.verb = verb; }


	/**
	 * The ID of the behaviour performed by the user. Each behaviour has a corresponding
	 * ID which can be used to track a behaviour through the API. This is an alternative
	 * to using the behaviour's verb. If verb and behaviourId are populated, the verb will
	 * take precedence.
	 * @return behaviourId
	 */
	public String getBehaviourId() { return behaviourId; }
	public void setBehaviourId( String behaviourId ) { this.behaviourId = behaviourId; }

	/**
	 * The metadata associated to the behaviour performed by the user.
	 * See the flexible behaviour page in the administration panel for more information.
	 * @return
	 */
	public Multimap<String,String> getMetadata() { return metadata; }
	public void setMetadata( Multimap<String,String> metadata ) { this.metadata = metadata; }
	public void addMetadata( String name, String value ) { metadata.put( name, value ); }
	public void clearMetadata() { metadata.clear(); }


	/**
	 * Any ad hoc points associated with this behaviour activity.
	 * The ad hoc points will be added to the points of the simple
	 * behaviour if it's status is OK. Otherwise the ad hoc points
	 * are added to the first matching flexible behaviour with a
	 * status of OK. If there are no behaviours with a status of
	 * OK, then the points are not added.
	 * @return
	 */
	public List<PointStoreDto> getAdhocPoints() { return adhocPoints; }
	public void setAdhocPoints( List<PointStoreDto> adhocPoints ) { this.adhocPoints = adhocPoints; }
	public void addAdhocPoint( PointStoreDto adhocPoint ) {
		if( adhocPoints == null ) {
			adhocPoints = new ArrayList<>();
		}
		adhocPoints.add( adhocPoint );
	}
}
