/*
 * Copyright (C) 2014-2025 Mambo Solutions Ltd.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package io.mambo.sdk.service.activity;

import java.util.List;

import io.mambo.sdk.http.HttpMethod;
import io.mambo.sdk.http.RequestOptions;
import io.mambo.sdk.http.ResponseType;
import io.mambo.sdk.http.api.ApiRequest;
import io.mambo.sdk.http.api.ApiRequestAdapter;
import io.mambo.sdk.service.activity.data.ActivityRequestData;
import io.mambo.sdk.service.activity.data.RejectActivityCriteriaRequestData;
import io.mambo.sdk.service.activity.data.RejectActivityRequestData;
import io.mambo.sdk.service.activity.model.ActivityDto;
import io.mambo.sdk.service.activity.param.ActivityBountyAwardParams;
import io.mambo.sdk.service.activity.param.ActivityBountyCancelParams;
import io.mambo.sdk.service.activity.param.ActivityCreateParams;
import io.mambo.sdk.service.activity.param.ActivityGetListByUserParams;
import io.mambo.sdk.service.activity.param.ActivityGetListParams;
import io.mambo.sdk.service.activity.param.ActivityGetParams;
import io.mambo.sdk.service.activity.param.ActivityProcessingStatusParams;
import io.mambo.sdk.service.activity.param.ActivityRejectBulkParams;
import io.mambo.sdk.service.activity.param.ActivityRejectParams;
import io.mambo.sdk.service.common.AbstractService;
import io.mambo.sdk.service.common.model.AggregateId;
import io.mambo.sdk.service.common.model.response.ProcessingStatusDto;
import io.mambo.sdk.service.common.model.response.RejectActivityBulkResponse;
import io.mambo.sdk.service.common.model.response.Status;

/**
 * The ActivitiesService class handles all Activity related requests to the
 * Mambo API.
 */
public class ActivitiesService extends AbstractService
{
	private static final String ACTIVITIES_SITE_URI = "/v1/{siteUrl}/activities";
	private static final String ACTIVITIES_SITE_URI_V2 = "/v2/{siteUrl}/activities";
	private static final String ACTIVITIES_USER_URI = "/v1/{siteUrl}/activities/{uuid}";
	private static final String ACTIVITIES_ID_URI = "/v1/activities/{activityId}";
	private static final String ACTIVITIES_REJECT_URI = ACTIVITIES_ID_URI + "/reject";
	private static final String ACTIVITIES_BULK_REJECT_URI = ACTIVITIES_SITE_URI + "/reject";
	private static final String ACTIVITIES_BOUNTY_CANCEL_URI = ACTIVITIES_ID_URI + "/bounty/cancel";
	private static final String ACTIVITIES_BOUNTY_AWARD_URI = ACTIVITIES_ID_URI + "/bounty/award/{uuid}";
	private static final String ACTIVITIES_STATUS_URI = "/v2/activities/{activityId}/status";


	public ActivitiesService( ApiRequestAdapter apiClient ) {
		super( apiClient );
	}


	/**
	 * This method is used to create activities. For more information on activities
	 * please refer to the POST Activities API documentation.
	 *
	 * @see ActivityRequestData
	 *
	 * @param siteUrl
	 *            The site to which the activity belongs
	 * @param data
	 *            The activity request data
	 * @return
	 */
	@Deprecated
	public AggregateId create( String siteUrl, ActivityRequestData data )
	{
		return create( siteUrl, data, RequestOptions.create() );
	}


	/**
	 * This method is used to create activities. For more information on activities
	 * please refer to the POST Activities API documentation.
	 *
	 * @see ActivityRequestData
	 *
	 * @param siteUrl
	 *            The site to which the activity belongs
	 * @param data
	 *            The activity request data
	 * @param requestOptions
	 *            The options to be used with this request
	 * @return
	 */
	@Deprecated
	public AggregateId create(
		String siteUrl,
		ActivityRequestData data,
		RequestOptions requestOptions )
	{
		return create( ActivityCreateParams.builder()
			.siteUrl( siteUrl )
			.build(), data, requestOptions );
	}


	/**
	 * This method is used to create activities. For more information on activities
	 * please refer to the POST Activities API documentation.
	 *
	 * @see ActivityRequestData
	 *
	 * @param params
	 *            The parameters required to create the activity
	 * @param data
	 *            The activity request data
	 * @param requestOptions
	 *            The options to be used with this request
	 * @return
	 */
	@Deprecated
	public AggregateId create(
		ActivityCreateParams params,
		ActivityRequestData data,
		RequestOptions requestOptions )
	{
		return apiClient().request( ApiRequest.builder()
			.apiPath( ACTIVITIES_SITE_URI )
			.responseClass( AggregateId.class )
			.method( HttpMethod.POST )
			.options( requestOptions )
			.requestData( data )
			.params( params )
			.build() );
	}


	/**
	 * This method is used to create activities. For more information on activities
	 * please refer to the POST Activities API documentation.
	 *
	 * @see ActivityRequestData
	 *
	 * @param siteUrl
	 *            The site to which the activity belongs
	 * @param data
	 *            The activity request data
	 * @return
	 */
	public AggregateId createAsync( String siteUrl, ActivityRequestData data )
	{
		return createAsync( siteUrl, data, RequestOptions.create() );
	}


	/**
	 * This method is used to create activities. For more information on activities
	 * please refer to the POST Activities API documentation.
	 *
	 * @see ActivityRequestData
	 *
	 * @param siteUrl
	 *            The site to which the activity belongs
	 * @param data
	 *            The activity request data
	 * @param requestOptions
	 *            The options to be used with this request
	 * @return
	 */
	public AggregateId createAsync(
		String siteUrl,
		ActivityRequestData data,
		RequestOptions requestOptions )
	{
		return createAsync( ActivityCreateParams.builder()
			.siteUrl( siteUrl )
			.build(), data, requestOptions );
	}


	/**
	 * This method is used to create activities. For more information on activities
	 * please refer to the POST Activities API documentation.
	 *
	 * @see ActivityRequestData
	 *
	 * @param params
	 *            The parameters required to create the activity
	 * @param data
	 *            The activity request data
	 * @param requestOptions
	 *            The options to be used with this request
	 * @return
	 */
	public AggregateId createAsync(
		ActivityCreateParams params,
		ActivityRequestData data,
		RequestOptions requestOptions )
	{
		return apiClient().request( ApiRequest.builder()
			.apiPath( ACTIVITIES_SITE_URI_V2 )
			.responseClass( AggregateId.class )
			.method( HttpMethod.POST )
			.options( requestOptions )
			.requestData( data )
			.params( params )
			.build() );
	}


	/**
	 * Reject an existing activity by ID
	 *
	 * @param id
	 *            The ID of the activity to reject
	 * @param data
	 *            The data {@link RejectActivityRequestData}
	 * @return
	 */
	public Status reject( String activityId, RejectActivityRequestData data )
	{
		return reject( ActivityRejectParams.builder()
			.activityId( activityId )
			.build(), data, RequestOptions.create() );
	}


	/**
	 * Reject an existing activity by ID
	 *
	 * @param params
	 *            The parameters required to reject an activity
	 * @param data
	 *            The data {@link RejectActivityRequestData}
	 * @return
	 */
	public Status reject( ActivityRejectParams params, RejectActivityRequestData data )
	{
		return reject( params, data, RequestOptions.create() );
	}


	/**
	 * Reject an existing activity by ID
	 *
	 * @param params
	 *            The parameters required to reject an activity
	 * @param data
	 *            The data {@link RejectActivityRequestData}
	 * @param requestOptions
	 *            The options to be used with this request
	 * @return
	 */
	public Status reject(
		ActivityRejectParams params,
		RejectActivityRequestData data,
		RequestOptions requestOptions )
	{
		return apiClient().request( ApiRequest.builder()
			.apiPath( ACTIVITIES_REJECT_URI )
			.responseClass( Status.class )
			.method( HttpMethod.POST )
			.options( requestOptions )
			.requestData( data )
			.params( params )
			.build() );
	}


	/**
	 * Reject activities in bulk by criteria and site
	 *
	 * @param siteUrl
	 *            The site to which the activities belongs
	 * @param data
	 *            The data {@link RejectActivityCriteriaRequestData}
	 * @return
	 */
	public RejectActivityBulkResponse rejectBulk(
		String siteUrl,
		RejectActivityCriteriaRequestData data )
	{
		return rejectBulk( ActivityRejectBulkParams.builder()
			.siteUrl( siteUrl )
			.build(), data, RequestOptions.create() );
	}


	/**
	 * Reject activities in bulk by criteria and site
	 *
	 * @param params
	 *            The parameters required to reject the activities
	 * @param data
	 *            The data {@link RejectActivityCriteriaRequestData}
	 * @return
	 */
	public RejectActivityBulkResponse rejectBulk(
		ActivityRejectBulkParams params,
		RejectActivityCriteriaRequestData data )
	{
		return rejectBulk( params, data, RequestOptions.create() );
	}


	/**
	 * Reject activities in bulk by criteria and site
	 *
	 * @param params
	 *            The parameters required to reject the activities
	 * @param data
	 *            The data {@link RejectActivityCriteriaRequestData}
	 * @param requestOptions
	 *            The options to be used with this request
	 * @return
	 */
	public RejectActivityBulkResponse rejectBulk(
		ActivityRejectBulkParams params,
		RejectActivityCriteriaRequestData data,
		RequestOptions requestOptions )
	{
		return apiClient().request( ApiRequest.builder()
			.apiPath( ACTIVITIES_BULK_REJECT_URI )
			.responseClass( RejectActivityBulkResponse.class )
			.method( HttpMethod.POST )
			.options( requestOptions )
			.requestData( data )
			.params( params )
			.build() );
	}


	/**
	 * Cancel an existing bounty activity by ID
	 *
	 * @param id
	 *            The ID of the bounty activity to cancel
	 * @return
	 */
	public Status bountyCancel( String activityId )
	{
		return bountyCancel( ActivityBountyCancelParams.builder()
			.activityId( activityId )
			.build() );
	}


	/**
	 * Cancel an existing bounty activity by ID
	 *
	 * @param params
	 *            The parameters required to cancel the bounty
	 * @return
	 */
	public Status bountyCancel( ActivityBountyCancelParams params )
	{
		return bountyCancel( params, RequestOptions.create() );
	}


	/**
	 * Cancel an existing bounty activity by ID
	 *
	 * @param params
	 *            The parameters required to cancel the bounty
	 * @param requestOptions
	 *            The options to be used with this request
	 * @return
	 */
	public Status bountyCancel( ActivityBountyCancelParams params, RequestOptions requestOptions )
	{
		return apiClient().request( ApiRequest.builder()
			.apiPath( ACTIVITIES_BOUNTY_CANCEL_URI )
			.responseClass( Status.class )
			.method( HttpMethod.POST )
			.options( requestOptions )
			.params( params )
			.build() );
	}


	/**
	 * Award an existing bounty activity by ID to the user specified
	 *
	 * @param activityId
	 *            The ID of the bounty activity to award
	 * @param uuid
	 *            The UUID of the user to whom the bounty should be awarded
	 * @return
	 */
	public Status bountyAward( String activityId, String uuid )
	{
		return bountyAward( ActivityBountyAwardParams.builder()
			.activityId( activityId )
			.uuid( uuid )
			.build() );
	}


	/**
	 * Award an existing bounty activity by ID to the user specified
	 *
	 * @param params
	 *            The parameters required to award the bounty
	 * @return
	 */
	public Status bountyAward( ActivityBountyAwardParams params )
	{
		return bountyAward( params, RequestOptions.create() );
	}


	/**
	 * Award an existing bounty activity by ID to the user specified
	 *
	 * @param params
	 *            The parameters required to award the bounty
	 * @param requestOptions
	 *            The options to be used with this request
	 * @return
	 */
	public Status bountyAward( ActivityBountyAwardParams params, RequestOptions requestOptions )
	{
		return apiClient().request( ApiRequest.builder()
			.apiPath( ACTIVITIES_BOUNTY_AWARD_URI )
			.responseClass( Status.class )
			.method( HttpMethod.POST )
			.options( requestOptions )
			.params( params )
			.build() );
	}


	/**
	 * Get an activity by ID
	 *
	 * @param activityId
	 *            The ID of the activity to retrieve
	 * @return
	 */
	public ActivityDto get( String activityId )
	{
		return get( ActivityGetParams.builder().activityId( activityId ).build() );
	}


	/**
	 * Get an activity by ID
	 *
	 * @param params
	 *            The parameters required to retrieve the activity
	 * @return
	 */
	public ActivityDto get( ActivityGetParams params )
	{
		return get( params, RequestOptions.create() );
	}


	/**
	 * Get an activity by ID
	 *
	 * @param params
	 *            The parameters required to retrieve the activity
	 * @param requestOptions
	 *            The options to be used with this request
	 * @return
	 */
	public ActivityDto get( ActivityGetParams params, RequestOptions requestOptions )
	{
		return apiClient().request( ApiRequest.builder()
			.apiPath( ACTIVITIES_ID_URI )
			.responseClass( ActivityDto.class )
			.method( HttpMethod.GET )
			.options( requestOptions )
			.params( params )
			.build() );
	}


	/**
	 * Get the list of activities for the specified site starting from the page
	 * specified and returning the number of activities specified by count
	 *
	 * @param siteUrl
	 *            The site from which to retrieve the activities
	 * @return
	 */
	public List<ActivityDto> getActivities( String siteUrl )
	{
		return getActivities( ActivityGetListParams.builder()
			.siteUrl( siteUrl )
			.build() );
	}


	/**
	 * Get the list of activities for the specified site starting from the page
	 * specified and returning the number of activities specified by count
	 *
	 * @param params
	 *            The parameters required to retrieve the list of activities
	 * @return
	 */
	public List<ActivityDto> getActivities( ActivityGetListParams params )
	{
		return getActivities( params, RequestOptions.create() );
	}


	/**
	 * Get the list of activities for the specified site starting from the page
	 * specified and returning the number of activities specified by count
	 *
	 * @param params
	 *            The parameters required to retrieve the list of activities
	 * @param requestOptions
	 *            The options to be used with this request
	 * @return
	 */
	public List<ActivityDto> getActivities(
		ActivityGetListParams params,
		RequestOptions requestOptions )
	{
		return apiClient().request( ApiRequest.builder()
			.apiPath( ACTIVITIES_SITE_URI )
			.responseClass( ActivityDto.class )
			.responseType( ResponseType.LIST )
			.method( HttpMethod.GET )
			.options( requestOptions )
			.params( params )
			.build() );
	}


	/**
	 * Get the list of activities for the specified user and site starting from the
	 * page specified and returning the number of activities specified by count
	 *
	 * @param siteUrl
	 *            The site from which to retrieve the activities
	 * @param uuid
	 *            The uuid for which to retrieve the activities
	 * @return
	 */
	public List<ActivityDto> getActivitiesByUser( String siteUrl, String uuid )
	{
		return getActivitiesByUser( ActivityGetListByUserParams.builder()
			.siteUrl( siteUrl )
			.uuid( uuid )
			.build(), RequestOptions.create() );
	}


	/**
	 * Get the list of activities for the specified user and site starting from the
	 * page specified and returning the number of activities specified by count
	 *
	 * @param params
	 *            The parameters required to retrieve the list of activities
	 * @return
	 */
	public List<ActivityDto> getActivitiesByUser( ActivityGetListByUserParams params )
	{
		return getActivitiesByUser( params, RequestOptions.create() );
	}


	/**
	 * Get the list of activities for the specified user and site starting from the
	 * page specified and returning the number of activities specified by count
	 *
	 * @param params
	 *            The parameters required to retrieve the list of activities
	 * @param requestOptions
	 *            The options to be used with this request
	 * @return
	 */
	public List<ActivityDto> getActivitiesByUser(
		ActivityGetListByUserParams params,
		RequestOptions requestOptions )
	{
		return apiClient().request( ApiRequest.builder()
			.apiPath( ACTIVITIES_USER_URI )
			.responseClass( ActivityDto.class )
			.responseType( ResponseType.LIST )
			.method( HttpMethod.GET )
			.options( requestOptions )
			.params( params )
			.build() );
	}


	/**
	 * Return the processing status of a submitted Activity
	 *
	 * @param id
	 *            The id of the activity to retrieve
	 * @return
	 */
	public ProcessingStatusDto getProcessingStatus( String activityId )
	{
		return getProcessingStatus( ActivityProcessingStatusParams.builder()
			.activityId( activityId )
			.build() );
	}


	/**
	 * Return the processing status of a submitted Activity
	 *
	 * @param params
	 *            The parameters required to retrieve the processing status of the
	 *            activity
	 * @return
	 */
	public ProcessingStatusDto getProcessingStatus( ActivityProcessingStatusParams params )
	{
		return getProcessingStatus( params, RequestOptions.create() );
	}


	/**
	 * Return the processing status of a submitted Activity
	 *
	 * @param params
	 *            The parameters required to retrieve the processing status of the
	 *            activity
	 * @param requestOptions
	 *            The options to be used with this request
	 * @return
	 */
	public ProcessingStatusDto getProcessingStatus(
		ActivityProcessingStatusParams params,
		RequestOptions requestOptions )
	{
		return apiClient().request( ApiRequest.builder()
			.apiPath( ACTIVITIES_STATUS_URI )
			.responseClass( ProcessingStatusDto.class )
			.method( HttpMethod.GET )
			.options( requestOptions )
			.params( params )
			.build() );
	}


	/**
	 * Return an empty {@link ActivityRequestData} object
	 *
	 * @return
	 */
	public ActivityRequestData newActivityRequestData()
	{
		return new ActivityRequestData();
	}


	/**
	 * Return an empty {@link RejectActivityRequestData} object
	 *
	 * @return
	 */
	public RejectActivityRequestData newRejectActivityRequestData()
	{
		return new RejectActivityRequestData();
	}


	/**
	 * Return an empty {@link RejectActivityCriteriaRequestData} object
	 *
	 * @return
	 */
	public RejectActivityCriteriaRequestData newRejectActivityCriteriaRequestData()
	{
		return new RejectActivityCriteriaRequestData();
	}
}
