/*
 * Copyright (C) 2014-2025 Mambo Solutions Ltd.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package io.mambo.sdk.http.client;

import java.nio.charset.StandardCharsets;
import java.util.stream.Collectors;

import io.mambo.sdk.ClientConfiguration;
import io.mambo.sdk.http.HttpHeaders;
import io.mambo.sdk.http.HttpRequest;
import io.mambo.sdk.http.HttpResponse;
import io.mambo.sdk.utils.MamboLogger;

public class HttpClientLogger
{
	private final ClientConfiguration configuration;
	private final MamboLogger logger;


	public HttpClientLogger( ClientConfiguration configuration )
	{
		this.configuration = configuration;
		this.logger = new MamboLogger( configuration );
	}


	public void log( String logMessage )
	{
		logger.log( logMessage );
	}


	public void logRequest( int requestId, HttpRequest request )
	{
		if( !configuration.hasLogging() ) {
			return;
		}

		StringBuilder builder = new StringBuilder();
		builder.append( String.format( "%s > %s %s%n", requestId, request.method().name(), request.url() ) );
		addHeaders( requestId, request.headers(), ">", builder );

		if( request.hasContent() ) {
			String contentAsString = new String( request.content().contentByteArray(), StandardCharsets.UTF_8 );
			builder.append( String.format( "%s > %n%s%n", requestId, contentAsString.trim() ) );
		}

		logger.log( "MamboClient sent a request on thread {}\n{}",
			Thread.currentThread().getName(), builder );
	}


	private void addHeaders( int requestId, HttpHeaders headers, String separator, StringBuilder builder )
	{
		builder.append( headers.entrySet().stream()
				.map( entry -> requestId + " " + separator + " " + entry.getKey() + ": " + String.join( ", ", entry.getValue() ) )
				.collect( Collectors.joining( System.lineSeparator() ) ) )
			.append( System.lineSeparator() );
	}


	public void logResponse( int requestId, HttpResponse response )
	{
		if( !configuration.hasLogging() ) {
			return;
		}

		StringBuilder builder = new StringBuilder();
		builder.append( String.format( "%s < %s%n", requestId, response.code() ) );
		addHeaders( requestId, response.headers(), "<", builder );

		if( response.hasContent() ) {
			builder.append( String.format( "%s < %n%s%n", requestId, response.content() ) );
		}

		logger.log( "MamboClient received a response on thread {}\n{}",
			Thread.currentThread().getName(), builder );
	}
}
