/*
 * Copyright (C) 2014-2025 Mambo Solutions Ltd.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package io.mambo.sdk.http.client;

import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.net.HttpURLConnection;
import java.net.Proxy;
import java.util.List;
import java.util.Map;
import java.util.concurrent.atomic.AtomicInteger;

import javax.net.ssl.HostnameVerifier;
import javax.net.ssl.HttpsURLConnection;
import javax.net.ssl.SSLContext;
import javax.net.ssl.TrustManager;

import io.mambo.sdk.ClientConfiguration;
import io.mambo.sdk.exception.ApiConnectionException;
import io.mambo.sdk.exception.MamboException;
import io.mambo.sdk.http.HttpHeaders;
import io.mambo.sdk.http.HttpRequest;
import io.mambo.sdk.http.HttpResponse;
import io.mambo.sdk.http.HttpStatus;
import io.mambo.sdk.http.NoOpTrustManager;

public class DefaultHttpClient implements HttpClient
{
	private static final AtomicInteger requestCount = new AtomicInteger( 1 );

	private final HttpClientLogger logger;


	public DefaultHttpClient( ClientConfiguration configuration ) {
		this.logger = new HttpClientLogger( configuration );
	}


	@Override
	public HttpResponse request( HttpRequest request )
	{
		try {
			int requestId = requestCount.incrementAndGet();
			logger.logRequest( requestId, request );
			HttpResponse response = executeRequest( request );
			logger.logResponse( requestId, response );
			return response;
		}
		catch( IOException exception ) {
			throw new ApiConnectionException( exception );
		}
	}


	private HttpResponse executeRequest( HttpRequest request ) throws IOException
	{
		HttpURLConnection connection = buildConnection( request );

		HttpStatus status = HttpStatus.valueOf( connection.getResponseCode() );
		HttpHeaders headers = HttpHeaders.of( connection.getHeaderFields() );
		InputStream bodyStream = responseBody( connection, status );

		return new HttpResponse( status, headers, bodyStream );
	}


	private HttpURLConnection buildConnection( HttpRequest request ) throws IOException
	{
		HttpURLConnection connection = openConnection( request );
		connection.setConnectTimeout( request.options().connectionTimeout() );
		connection.setReadTimeout( request.options().readTimeout() );
		connection.setRequestMethod( request.method().name() );
		connection.setUseCaches( false );
		addHeaders( connection, request );
		addContent( connection, request );
		return connection;
	}


	private HttpURLConnection openConnection( HttpRequest request ) throws IOException
	{
		if( request.options().hasProxy() ) {
			return openProxyConnection( request );
		}
		return ( HttpURLConnection ) request.url().openConnection();
	}


	private HttpURLConnection openProxyConnection( HttpRequest request ) throws IOException
	{
		Proxy proxy = request.options().proxy();
		HttpURLConnection connection = ( HttpURLConnection ) request.url().openConnection( proxy );
		addProxyAuthentication( request, connection );
		return connection;
	}


	private void addProxyAuthentication( HttpRequest request, HttpURLConnection connection )
	{
		if( request.options().hasProxyCredentials() ) {
			String credentials = request.options().encodedProxyCredentials();
			connection.setRequestProperty( "Proxy-Authorization", "Basic " + credentials );
		}
	}


	private void addHeaders( HttpURLConnection connection, HttpRequest request )
	{
		for( Map.Entry<String, List<String>> entry : request.headers().entrySet() ) {
			connection.setRequestProperty( entry.getKey(), String.join( ", ", entry.getValue() ) );
		}
	}


	private void addContent( HttpURLConnection connection, HttpRequest request ) throws IOException
	{
		if( !request.hasContent() ) {
			return;
		}

		connection.setDoOutput( true );
		connection.setRequestProperty( "Content-Type", request.content().contentType() );
		try( OutputStream output = connection.getOutputStream() ) {
			output.write( request.content().contentByteArray() );
		}
	}


	private InputStream responseBody( HttpURLConnection connection, HttpStatus status ) throws IOException
	{
		return ( status.is2xxSuccessful() )
			? connection.getInputStream()
			: connection.getErrorStream();
	}


	@Override
	public void disableSslVerification()
	{
		try {
			HostnameVerifier verifier = ( urlHostName, session ) -> true;
			TrustManager[] trustAllCerts = new TrustManager[] { new NoOpTrustManager() };
			SSLContext sslContext = SSLContext.getInstance( "TLS" );
			sslContext.init( null, trustAllCerts, new java.security.SecureRandom() );
			HttpsURLConnection.setDefaultSSLSocketFactory( sslContext.getSocketFactory() );
			HttpsURLConnection.setDefaultHostnameVerifier( verifier );
			SSLContext.setDefault( sslContext );
			logger.log( "Disabled SSL verification" );
		}
		catch( Exception exception ) {
			throw new MamboException( "Error disabling SSL verification.", exception );
		}
	}
}
