/*
 * Copyright (C) 2014-2025 Mambo Solutions Ltd.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package io.mambo.sdk.http.authenticator;

import static io.mambo.sdk.utils.StringUtils.isNotEmpty;

import io.mambo.sdk.exception.MamboApiException;
import io.mambo.sdk.exception.MamboExceptionFactory;
import io.mambo.sdk.http.HttpContent;
import io.mambo.sdk.http.HttpHeaders;
import io.mambo.sdk.http.HttpMethod;
import io.mambo.sdk.http.HttpParams;
import io.mambo.sdk.http.HttpRequest;
import io.mambo.sdk.http.HttpResponse;
import io.mambo.sdk.http.RequestOptions;
import io.mambo.sdk.http.client.HttpClient;
import io.mambo.sdk.utils.JsonUtils;

public class OAuthAuthenticator implements Authenticator
{
	private static final MamboExceptionFactory exceptionFactory = new MamboExceptionFactory();

	private static final String ACCESS_TOKEN_URL = "/oauth/token";

	private final HttpClient httpClient;
	private final MamboCredentials credentials;

	private String oauthToken = null;


	public OAuthAuthenticator(
		MamboCredentials credentials,
		HttpClient httpClient )
	{
		this.credentials = credentials;
		this.httpClient = httpClient;
	}


	@Override
	public void authenticate( HttpHeaders headers, RequestOptions options )
	{
		String bearerToken = fetchOAuthToken( headers, options );
		headers.addHeader( "Authorization", "Bearer " + bearerToken );
	}


	private String fetchOAuthToken( HttpHeaders headers, RequestOptions options )
	{
		if( isNotEmpty( oauthToken ) ) {
			return oauthToken;
		}

		refreshOAuthToken( headers, options );
		return oauthToken;
	}


	private void refreshOAuthToken( HttpHeaders headers, RequestOptions options )
	{
		HttpRequest request = toHttpRequest( headers, options );
		HttpResponse response = httpClient.request( request );

		if( !response.code().is2xxSuccessful() ) {
			throw createException( response );
		}

		oauthToken = toAccessToken( response );
	}


	private String toAccessToken( HttpResponse response )
	{
		AccessToken accessToken = JsonUtils.deserialize( response.content(), AccessToken.class );
		return accessToken.getAccessToken();
	}


	private HttpRequest toHttpRequest( HttpHeaders headers, RequestOptions options )
	{
		return new HttpRequest(
			buildOAuthUrl( options ),
			HttpMethod.POST,
			buildBasicAuthHeaders( headers ),
			HttpParams.empty(),
			HttpContent.form( "grant_type=client_credentials" ),
			options );
	}


	private String buildOAuthUrl( RequestOptions options )
	{
		StringBuilder urlBuilder = new StringBuilder();
		urlBuilder.append( options.serverBaseUrl() );
		urlBuilder.append( ACCESS_TOKEN_URL );
		return urlBuilder.toString();
	}


	private HttpHeaders buildBasicAuthHeaders( HttpHeaders headers )
	{
		return HttpHeaders.of( headers )
			.addHeader( "Authorization", "Basic " + credentials.encodedCredentials() );
	}


	private MamboApiException createException( HttpResponse response )
	{
		return exceptionFactory.createMamboApiException( response.content(), response.code() );
	}


	@Override
	public void invalidate()
	{
		oauthToken = null;
	}
}
