/*
 * Copyright (C) 2014-2025 Mambo Solutions Ltd.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package io.mambo.sdk.http.api;

import static io.mambo.sdk.utils.Preconditions.checkNotNull;

import io.mambo.sdk.ClientConfiguration;
import io.mambo.sdk.Version;
import io.mambo.sdk.exception.MamboApiException;
import io.mambo.sdk.exception.MamboExceptionFactory;
import io.mambo.sdk.exception.oauth.InvalidTokenException;
import io.mambo.sdk.http.HttpContent;
import io.mambo.sdk.http.HttpHeaders;
import io.mambo.sdk.http.HttpParams;
import io.mambo.sdk.http.HttpRequest;
import io.mambo.sdk.http.HttpResponse;
import io.mambo.sdk.http.RequestOptions;
import io.mambo.sdk.http.authenticator.Authenticator;
import io.mambo.sdk.http.authenticator.MamboCredentials;
import io.mambo.sdk.http.authenticator.OAuthAuthenticator;
import io.mambo.sdk.http.client.DefaultHttpClient;
import io.mambo.sdk.http.client.HttpClient;
import io.mambo.sdk.utils.JsonUtils;
import io.mambo.sdk.utils.ObjectUtils;

/**
 * Responsible for mapping {@link ApiRequest} to and from the {@link DefaultHttpClient}
 */
public class DefaultApiRequestAdapter implements ApiRequestAdapter
{
	private static final MamboExceptionFactory exceptionFactory = new MamboExceptionFactory();

	private static final String API_BASE_PATH = "/api";

	private final Version version = new Version();

	private final HttpClient httpClient;
	private final Authenticator authenticator;
	private final ClientConfiguration configuration;


	public DefaultApiRequestAdapter(
		MamboCredentials credentials,
		ClientConfiguration configuration )
	{
		this( credentials, configuration, null, null );
	}


	public DefaultApiRequestAdapter(
		MamboCredentials credentials,
		ClientConfiguration configuration,
		HttpClient client,
		Authenticator authenticator )
	{
		checkNotNull( credentials, "credentials must not be null" );
		checkNotNull( configuration, "configuration must not be null" );

		this.configuration = configuration;
		this.httpClient = buildHttpClient( configuration, client );
		this.authenticator = buildAuthenticator( credentials, authenticator );
		sslConfiguration();
	}


	private HttpClient buildHttpClient( ClientConfiguration configuration, HttpClient client )
	{
		return ObjectUtils.getOrDefault( client, new DefaultHttpClient( configuration ) );
	}


	private Authenticator buildAuthenticator( MamboCredentials credentials, Authenticator authenticator )
	{
		return ObjectUtils.getOrDefault( authenticator, new OAuthAuthenticator( credentials, httpClient ) );
	}


	private void sslConfiguration()
	{
		if( !configuration.hasSslVerification() ) {
			httpClient.disableSslVerification();
		}
	}


	@Override
	public String getVersion()
	{
		return version.getVersion();
	}


	@Override
	public <T> T request( ApiRequest apiRequest )
	{
		HttpRequest request = toHttpRequest( apiRequest );
		HttpResponse response = httpClient.request( request );

		if( !response.code().is2xxSuccessful() ) {
			return handleError( apiRequest, response );
		}

		return deserializeResponse( apiRequest, response );
	}


	private HttpRequest toHttpRequest( ApiRequest apiRequest )
	{
		RequestOptions options = mergeDefaultOptions( apiRequest.options() );

		return new HttpRequest(
			buildResourceUrl( apiRequest, options ),
			apiRequest.method(),
			buildBaseHeaders( options ),
			HttpParams.of( apiRequest.params() ),
			HttpContent.of( apiRequest.requestData() ),
			options );
	}


	private String buildResourceUrl( ApiRequest apiRequest, RequestOptions options )
	{
		StringBuilder urlBuilder = new StringBuilder();
		urlBuilder.append( options.serverBaseUrl() );
		urlBuilder.append( API_BASE_PATH );
		urlBuilder.append( apiRequest.apiPath() );
		return urlBuilder.toString();
	}


	private HttpHeaders buildBaseHeaders( RequestOptions options )
	{
		HttpHeaders headers = new HttpHeaders()
			.addHeader( "User-Agent", userAgent() )
			.addHeader( "Accept-Version", getVersion() )
			.addHeader( "Accept", "application/json" );
		authenticator.authenticate( headers, options );
		return headers;
	}


	private String userAgent()
	{
		return "Mambo/" + getVersion() + "; Java " + System.getProperty( "java.version" );
	}


	private RequestOptions mergeDefaultOptions( RequestOptions requestOptions )
	{
		return requestOptions.merge( configuration );
	}


	private <T> T handleError( ApiRequest apiRequest, HttpResponse response )
	{
		MamboApiException exception = createException( response );

		if( exception instanceof InvalidTokenException ) {
			authenticator.invalidate();
			return request( apiRequest );
		}

		throw exception;
	}


	private MamboApiException createException( HttpResponse response )
	{
		return exceptionFactory.createMamboApiException( response.content(), response.code() );
	}


	private <T> T deserializeResponse( ApiRequest apiRequest, HttpResponse response )
	{
		return JsonUtils.deserialize(
			response.content(),
			apiRequest.responseClass(),
			apiRequest.responseType() );
	}
}
