/*
 * Copyright (C) 2014-2025 Mambo Solutions Ltd.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package io.mambo.sdk.http.api;

import static io.mambo.sdk.utils.Preconditions.checkNotNull;

import io.mambo.sdk.http.HttpMethod;
import io.mambo.sdk.http.RequestOptions;
import io.mambo.sdk.http.ResponseType;
import io.mambo.sdk.service.common.data.RequestData;
import io.mambo.sdk.service.common.param.ServiceParams;
import io.mambo.sdk.utils.Multimap;

/**
 * Represents an API request coming from a Mambo service
 */
public class ApiRequest
{
    private final String apiPath;
    private final Class<?> responseClass;
    private final ResponseType responseType;
    private final HttpMethod method;
    private final RequestData requestData;
    private final RequestOptions options;
    private final ServiceParams params;


	private ApiRequest( Builder builder ) {
		this.apiPath = builder.apiPath;
		this.responseClass = builder.responseClass;
		this.method = builder.method;
		this.requestData = builder.requestData;
		this.responseType = builder.responseType;
		this.options = builder.options;
		this.params = builder.params;
	}


	public static Builder builder()
	{
		return new Builder();
	}


    public String apiPath() { return apiPath; }
    public Class<?> responseClass() { return responseClass; }
    public ResponseType responseType() { return responseType; }
    public HttpMethod method() { return method; }
    public RequestData requestData() { return requestData; }
    public RequestOptions options() { return options; }

    public Multimap<String, String> params() {
    	if( params == null ) {
    		return new Multimap<>();
    	}
    	return params.asMap();
    }


	public static class Builder
	{
		private String apiPath;
		private Class<?> responseClass;
		private ResponseType responseType = ResponseType.OBJECT;
		private HttpMethod method;
		private RequestData requestData;
		private RequestOptions options = RequestOptions.create();
	    private ServiceParams params;


		private Builder() {}


		public Builder apiPath( String apiPath )
		{
			this.apiPath = apiPath;
			return this;
		}


		public Builder responseClass( Class<?> responseClass )
		{
			this.responseClass = responseClass;
			return this;
		}


		public Builder responseType( ResponseType responseType )
		{
			this.responseType = responseType;
			return this;
		}


		public Builder method( HttpMethod method )
		{
			this.method = method;
			return this;
		}


		public Builder requestData( RequestData requestData )
		{
			this.requestData = requestData;
			return this;
		}


		public Builder options( RequestOptions options )
		{
			this.options = options;
			return this;
		}


		public Builder params( ServiceParams params )
		{
			this.params = params;
			return this;
		}


		public ApiRequest build()
		{
			validateRequest();
			return new ApiRequest( this );
		}


		private void validateRequest()
		{
			checkNotNull( apiPath, "apiPath must not be null" );
			checkNotNull( responseClass, "responseClass must not be null" );
			checkNotNull( responseType, "responseType must not be null" );
			checkNotNull( method, "method must not be null" );
		}
	}
}
