/*
 * Copyright (C) 2014-2025 Mambo Solutions Ltd.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package io.mambo.sdk.http;

import static io.mambo.sdk.utils.Preconditions.checkArgument;
import static io.mambo.sdk.utils.StringUtils.isNotBlank;
import static io.mambo.sdk.utils.StringUtils.startsWith;

import java.net.Proxy;
import java.nio.charset.StandardCharsets;
import java.util.Base64;
import java.util.UUID;

import io.mambo.sdk.ClientConfiguration;
import io.mambo.sdk.utils.HttpUtils;
import io.mambo.sdk.utils.ObjectUtils;
import io.mambo.sdk.utils.StringUtils;
import lombok.ToString;

/**
 * Configurable options used in the HTTP request
 */
@ToString
public class RequestOptions
{
	private final String serverBaseUrl;
	private final String language;
	private final Integer readTimeoutMillis;
	private final Integer connectionTimeoutMillis;
	private final Proxy proxy;
	private final String proxyUsername;
	private final String proxyPassword;
	private final UUID idempotencyKey;


	private RequestOptions( Builder builder ) {
		this.serverBaseUrl = builder.serverBaseUrl;
		this.language = builder.language;
		this.readTimeoutMillis = builder.readTimeoutMillis;
		this.connectionTimeoutMillis = builder.connectionTimeoutMillis;
		this.proxy = builder.proxy;
		this.proxyUsername = builder.proxyUsername;
		this.proxyPassword = builder.proxyPassword;
		this.idempotencyKey = builder.idempotencyKey;
	}


	private RequestOptions(
		String serverBaseUrl,
		String language,
		Integer readTimeoutMillis,
		Integer connectionTimeoutMillis,
		Proxy proxy,
		String proxyUsername,
		String proxyPassword,
		UUID idempotencyKey )
	{
		this.serverBaseUrl = serverBaseUrl;
		this.language = language;
		this.readTimeoutMillis = readTimeoutMillis;
		this.connectionTimeoutMillis = connectionTimeoutMillis;
		this.proxy = proxy;
		this.proxyUsername = proxyUsername;
		this.proxyPassword = proxyPassword;
		this.idempotencyKey = idempotencyKey;
	}


	public static RequestOptions create()
	{
		return new RequestOptions( null, null, null, null, null, null, null, null );
	}


	public static Builder builder()
	{
		return new Builder();
	}


	public String serverBaseUrl()
	{
		return serverBaseUrl;
	}


	public String language()
	{
		return language;
	}


	public boolean hasProxy()
	{
		return proxy != null;
	}


	public Proxy proxy()
	{
		return proxy;
	}


	public boolean hasProxyCredentials()
	{
		return StringUtils.isNotBlank( proxyUsername ) &&
			StringUtils.isNotBlank( proxyPassword );
	}


	public String encodedProxyCredentials()
	{
		String auth = proxyUsername + ":" + proxyPassword;
		return Base64.getEncoder().encodeToString(
			auth.getBytes( StandardCharsets.UTF_8 ) );
	}


	public Integer connectionTimeout()
	{
		return connectionTimeoutMillis;
	}


	public Integer readTimeout()
	{
		return readTimeoutMillis;
	}


	public boolean hasIdempotencyKey()
	{
		return idempotencyKey != null;
	}


	public String idempotencyKey()
	{
		return hasIdempotencyKey() ? idempotencyKey.toString() : null;
	}


	public RequestOptions merge( ClientConfiguration configuration )
	{
		return new RequestOptions(
			ObjectUtils.getOrDefault( serverBaseUrl, configuration.getServerBaseUrl() ),
			ObjectUtils.getOrDefault( language, configuration.getLanguage() ),
			ObjectUtils.getOrDefault( readTimeoutMillis, configuration.getReadTimeoutMillis() ),
			ObjectUtils.getOrDefault( connectionTimeoutMillis, configuration.getConnectionTimeoutMillis() ),
			ObjectUtils.getOrDefault( proxy, configuration.getProxy() ),
			ObjectUtils.getOrDefault( proxyUsername, configuration.getProxyUsername() ),
			ObjectUtils.getOrDefault( proxyPassword, configuration.getProxyPassword() ),
			idempotencyKey );
	}


	public static class Builder
	{
		private String serverBaseUrl;
		private String language;
		private Integer readTimeoutMillis;
		private Integer connectionTimeoutMillis;
		private Proxy proxy;
		private String proxyUsername;
		private String proxyPassword;
		private UUID idempotencyKey;


		private Builder() {}


		/**
		 * This is used to set the Server base URL. For example:
		 * https://api.yourdomain.com
		 */
		public Builder serverBaseUrl( String serverBaseUrl )
		{
			checkArgument( isNotBlank( serverBaseUrl ), "serverBaseUrl must not be null or empty" );
			checkArgument( startsWith( serverBaseUrl, "http" ), "serverBaseUrl must include the protocol (e.g. https://)" );
			this.serverBaseUrl = StringUtils.removeEnd( serverBaseUrl, "/" );
			return this;
		}


		/**
		 * Set the Language to be used when calling the API
		 */
		public Builder language( String language )
		{
			checkArgument( isNotBlank( language ), "language must not be null or empty" );
			this.language = language;
			return this;
		}


		/**
		 * Proxy URI with port number (e.g. https://192.168.1.1:9400)
		 */
		public Builder proxyUri( String proxyUri )
		{
			checkArgument( isNotBlank( proxyUri ), "proxyUri must not be null or empty" );
			this.proxy = HttpUtils.buildProxy( proxyUri );
			return this;
		}


		/**
		 * Proxy to use for the connection
		 */
		public Builder proxy( Proxy proxy )
		{
			this.proxy = proxy;
			return this;
		}


		/**
		 * The proxy username, leave null if unauthenticated
		 */
		public Builder proxyUsername( String proxyUsername )
		{
			this.proxyUsername = proxyUsername;
			return this;
		}


		/**
		 * The proxy password, leave null if unauthenticated
		 */
		public Builder proxyPassword( String proxyPassword )
		{
			this.proxyPassword = proxyPassword;
			return this;
		}


		/**
		 * Set the timeout to read a response from the server. The default value is 0,
		 * which represents infinity.
		 */
		public Builder readTimeoutMillis( Integer readTimeoutMillis )
		{
			this.readTimeoutMillis = readTimeoutMillis;
			return this;
		}


		/**
		 * Set the timeout to connect to the server. The default value is 0, which
		 * represents infinity.
		 */
		public Builder connectionTimeoutMillis( Integer connectionTimeoutMillis )
		{
			this.connectionTimeoutMillis = connectionTimeoutMillis;
			return this;
		}


		/**
		 * Set the IdempotencyKey to be used for the request
		 */
		public Builder idempotencyKey( UUID idempotencyKey )
		{
			this.idempotencyKey = idempotencyKey;
			return this;
		}


		public RequestOptions build()
		{
			return new RequestOptions( this );
		}
	}
}