/*
 * Copyright (C) 2014-2025 Mambo Solutions Ltd.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package io.mambo.sdk.http;

import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.OutputStreamWriter;
import java.io.Writer;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;

import io.mambo.sdk.exception.MamboException;
import io.mambo.sdk.utils.JsonUtils;
import io.mambo.sdk.utils.StreamUtils;

public class MultipartBuilder
{
	private static final String CRLF = "\r\n";

	private final String boundary;
	private final ByteArrayOutputStream outputStream;
	private final Writer writer;


	public MultipartBuilder( String boundary ) {
		this.boundary = boundary;
		this.outputStream = new ByteArrayOutputStream();
		this.writer = new OutputStreamWriter( outputStream, StandardCharsets.UTF_8 );
	}


	public MultipartBuilder addFileField( String partName, File file )
	{
		try {
			writeLine( "--" + boundary );
			writeLine( String.format( "Content-Disposition: form-data; name=\"%s\"; filename=\"%s\"", partName, file.getName() ) );
			writeLine( "Content-Type: " + detectContentType( file ) );
			writeEmptyLine();
			StreamUtils.copy( new FileInputStream( file ), outputStream );
			writeEmptyLine();
			return this;
		}
		catch( IOException exception ) {
			throw new MamboException( "Failed to add multipart file content", exception );
		}
	}


	private void writeLine( String line ) throws IOException
	{
		writer.write( line );
		writer.write( CRLF );
		writer.flush();
	}


	private void writeEmptyLine() throws IOException
	{
		writeLine( "" );
	}


	private String detectContentType( File file )
	{
		try {
			String contentType = Files.probeContentType( file.toPath() );
			return contentType != null ? contentType : "application/octet-stream";
		}
		catch( IOException e ) {
			return "application/octet-stream";
		}
	}


	public MultipartBuilder addJsonField( String name, Object value )
	{
		try {
			writePartHeader( name, "application/json" );
			writeLine( JsonUtils.serialize( value ) );
			return this;
		}
		catch( IOException exception ) {
			throw new MamboException( "Failed to add multipart JSON field", exception );
		}
	}


	private void writePartHeader( String partName, String contentType ) throws IOException
	{
		writeLine( "--" + boundary );
		writeLine( String.format( "Content-Disposition: form-data; name=\"%s\"", partName ) );
		writeLine( "Content-Type: " + contentType );
		writeEmptyLine();
	}


	public byte[] build()
	{
		try {
			writeLine( "--" + boundary + "--" );
			flushAndCloseStreams();
			return outputStream.toByteArray();
		}
		catch( IOException exception ) {
			throw new MamboException( "Failed to build multipart content", exception );
		}
	}


	private void flushAndCloseStreams() throws IOException
	{
		writer.flush();
		writer.close();
		outputStream.flush();
		outputStream.close();
	}
}