/*
 * Copyright (C) 2014-2025 Mambo Solutions Ltd.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package io.mambo.sdk.http;

/**
 * Represent the HTTP response status code
 */
public enum HttpStatus
{
	// 1xx Informational

	CONTINUE( 100, Family.INFORMATIONAL, "Continue" ),
	SWITCHING_PROTOCOLS( 101, Family.INFORMATIONAL, "Switching Protocols" ),
	PROCESSING( 102, Family.INFORMATIONAL, "Processing" ),
	CHECKPOINT( 103, Family.INFORMATIONAL, "Checkpoint" ),

	// 2xx Success

	OK( 200, Family.SUCCESSFUL, "OK" ),
	CREATED( 201, Family.SUCCESSFUL, "Created" ),
	ACCEPTED( 202, Family.SUCCESSFUL, "Accepted" ),
	NON_AUTHORITATIVE_INFORMATION( 203, Family.SUCCESSFUL, "Non-Authoritative Information" ),
	NO_CONTENT( 204, Family.SUCCESSFUL, "No Content" ),
	RESET_CONTENT( 205, Family.SUCCESSFUL, "Reset Content" ),
	PARTIAL_CONTENT( 206, Family.SUCCESSFUL, "Partial Content" ),
	MULTI_STATUS( 207, Family.SUCCESSFUL, "Multi-Status" ),
	ALREADY_REPORTED( 208, Family.SUCCESSFUL, "Already Reported" ),
	IM_USED( 226, Family.SUCCESSFUL, "IM Used" ),

	// 3xx Redirection

	MULTIPLE_CHOICES( 300, Family.REDIRECTION, "Multiple Choices" ),
	MOVED_PERMANENTLY( 301, Family.REDIRECTION, "Moved Permanently" ),
	MOVED_TEMPORARILY( 302, Family.REDIRECTION, "Moved Temporarily" ),
	SEE_OTHER( 303, Family.REDIRECTION, "See Other" ),
	NOT_MODIFIED( 304, Family.REDIRECTION, "Not Modified" ),
	USE_PROXY( 305, Family.REDIRECTION, "Use Proxy" ),
	TEMPORARY_REDIRECT( 307, Family.REDIRECTION, "Temporary Redirect" ),
	PERMANENT_REDIRECT( 308, Family.REDIRECTION, "Permanent Redirect" ),

	// 4xx Client Error

	BAD_REQUEST( 400, Family.CLIENT_ERROR, "Bad Request" ),
	UNAUTHORIZED( 401, Family.CLIENT_ERROR, "Unauthorized" ),
	PAYMENT_REQUIRED( 402, Family.CLIENT_ERROR, "Payment Required" ),
	FORBIDDEN( 403, Family.CLIENT_ERROR, "Forbidden" ),
	NOT_FOUND( 404, Family.CLIENT_ERROR, "Not Found" ),
	METHOD_NOT_ALLOWED( 405, Family.CLIENT_ERROR, "Method Not Allowed" ),
	NOT_ACCEPTABLE( 406, Family.CLIENT_ERROR, "Not Acceptable" ),
	PROXY_AUTHENTICATION_REQUIRED( 407, Family.CLIENT_ERROR, "Proxy Authentication Required" ),
	REQUEST_TIMEOUT( 408, Family.CLIENT_ERROR, "Request Timeout" ),
	CONFLICT( 409, Family.CLIENT_ERROR, "Conflict" ),
	GONE( 410, Family.CLIENT_ERROR, "Gone" ),
	LENGTH_REQUIRED( 411, Family.CLIENT_ERROR, "Length Required" ),
	PRECONDITION_FAILED( 412, Family.CLIENT_ERROR, "Precondition Failed" ),
	REQUEST_ENTITY_TOO_LARGE( 413, Family.CLIENT_ERROR, "Request Entity Too Large" ),
	REQUEST_URI_TOO_LONG( 414, Family.CLIENT_ERROR, "Request-URI Too Long" ),
	UNSUPPORTED_MEDIA_TYPE( 415, Family.CLIENT_ERROR, "Unsupported Media Type" ),
	REQUESTED_RANGE_NOT_SATISFIABLE( 416, Family.CLIENT_ERROR, "Requested range not satisfiable" ),
	EXPECTATION_FAILED( 417, Family.CLIENT_ERROR, "Expectation Failed" ),
	INSUFFICIENT_SPACE_ON_RESOURCE( 419, Family.CLIENT_ERROR, "Insufficient Space On Resource" ),
	METHOD_FAILURE( 420, Family.CLIENT_ERROR, "Method Failure" ),
	UNPROCESSABLE_ENTITY( 422, Family.CLIENT_ERROR, "Unprocessable Entity" ),
	LOCKED( 423, Family.CLIENT_ERROR, "Locked" ),
	FAILED_DEPENDENCY( 424, Family.CLIENT_ERROR, "Failed Dependency" ),
	TOO_EARLY( 425, Family.CLIENT_ERROR, "Too Early" ),
	UPGRADE_REQUIRED( 426, Family.CLIENT_ERROR, "Upgrade Required" ),
	PRECONDITION_REQUIRED( 428, Family.CLIENT_ERROR, "Precondition Required" ),
	TOO_MANY_REQUESTS( 429, Family.CLIENT_ERROR, "Too Many Requests" ),
	REQUEST_HEADER_FIELDS_TOO_LARGE( 431, Family.CLIENT_ERROR, "Request Header Fields Too Large" ),
	UNAVAILABLE_FOR_LEGAL_REASONS( 451, Family.CLIENT_ERROR, "Unavailable For Legal Reasons" ),

	// 5xx Server Error

	INTERNAL_SERVER_ERROR( 500, Family.SERVER_ERROR, "Internal Server Error" ),
	NOT_IMPLEMENTED( 501, Family.SERVER_ERROR, "Not Implemented" ),
	BAD_GATEWAY( 502, Family.SERVER_ERROR, "Bad Gateway" ),
	SERVICE_UNAVAILABLE( 503, Family.SERVER_ERROR, "Service Unavailable" ),
	GATEWAY_TIMEOUT( 504, Family.SERVER_ERROR, "Gateway Timeout" ),
	HTTP_VERSION_NOT_SUPPORTED( 505, Family.SERVER_ERROR, "HTTP Version not supported" ),
	VARIANT_ALSO_NEGOTIATES( 506, Family.SERVER_ERROR, "Variant Also Negotiates" ),
	INSUFFICIENT_STORAGE( 507, Family.SERVER_ERROR, "Insufficient Storage" ),
	LOOP_DETECTED( 508, Family.SERVER_ERROR, "Loop Detected" ),
	BANDWIDTH_LIMIT_EXCEEDED( 509, Family.SERVER_ERROR, "Bandwidth Limit Exceeded" ),
	NOT_EXTENDED( 510, Family.SERVER_ERROR, "Not Extended" ),
	NETWORK_AUTHENTICATION_REQUIRED( 511, Family.SERVER_ERROR, "Network Authentication Required" );

	private static final HttpStatus[] cachedValues;

	static {
		cachedValues = values();
	}

	private final int value;
	private final Family series;
	private final String reasonPhrase;


	private HttpStatus( int value, Family series, String reasonPhrase ) {
		this.value = value;
		this.series = series;
		this.reasonPhrase = reasonPhrase;
	}


	/**
	 * Return the integer value of this status code.
	 */
	public int value()
	{
		return value;
	}


	/**
	 * Return the HTTP status family of this status code.
	 */
	public Family family()
	{
		return series;
	}


	/**
	 * Return the reason phrase of this status code.
	 */
	public String getReasonPhrase()
	{
		return reasonPhrase;
	}


	public boolean is1xxInformational()
	{
		return ( family() == Family.INFORMATIONAL );
	}


	public boolean is2xxSuccessful()
	{
		return ( family() == Family.SUCCESSFUL );
	}


	public boolean is3xxRedirection()
	{
		return ( family() == Family.REDIRECTION );
	}


	public boolean is4xxClientError()
	{
		return ( family() == Family.CLIENT_ERROR );
	}


	public boolean is5xxServerError()
	{
		return ( family() == Family.SERVER_ERROR );
	}


	@Override
	public String toString()
	{
		return value + " (" + reasonPhrase + ")";
	}


	public static HttpStatus valueOf( int statusCode )
	{
		for( HttpStatus status : cachedValues ) {
			if( status.value == statusCode ) {
				return status;
			}
		}

		throw new IllegalArgumentException( "No matching status for [" + statusCode + "]" );
	}


	public enum Family
	{
		INFORMATIONAL( 1 ), SUCCESSFUL( 2 ), REDIRECTION( 3 ), CLIENT_ERROR( 4 ), SERVER_ERROR( 5 );

		private final int value;


		private Family( int value ) {
			this.value = value;
		}


		public int value()
		{
			return value;
		}


		public static Family valueOf( int statusCode )
		{
			int seriesCode = statusCode / 100;
			for( Family series : values() ) {
				if( series.value == seriesCode ) {
					return series;
				}
			}

			throw new IllegalArgumentException( "No matching family for [" + statusCode + "]" );
		}
	}

}
