/*
 * Copyright (C) 2014-2025 Mambo Solutions Ltd.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package io.mambo.sdk.http;

import java.io.IOException;
import java.net.URL;

import io.mambo.sdk.exception.MamboException;

/**
 * Information required to make an HTTP Request
 */
public class HttpRequest
{
	private final URL resourceUrl;
	private final HttpMethod method;
	private final HttpContent content;
	private final HttpHeaders headers;
	private final HttpParams parameters;
	private final RequestOptions options;


	public HttpRequest(
		String resourceUrl,
		HttpMethod method,
		HttpHeaders headers,
		HttpParams parameters,
		HttpContent content,
		RequestOptions options )
	{
		this.method = method;
		this.content = content;
		this.parameters = parameters;
		this.options = options;
		this.resourceUrl = buildUrl( resourceUrl, parameters );
		this.headers = builderHeaders( headers, content, options );
	}


	private URL buildUrl(
		String resourceUrl,
		HttpParams parameters )
	{
		try {
			StringBuilder urlBuilder = new StringBuilder();
			urlBuilder.append( parameters.resolvePath( resourceUrl ) );
			urlBuilder.append( parameters.buildQueryString() );
			return new URL( urlBuilder.toString() );
		}
		catch( IOException exception ) {
			throw new MamboException( "Error building request URL", exception );
		}
	}


	private HttpHeaders builderHeaders(
		HttpHeaders headers,
		HttpContent content,
		RequestOptions options )
	{
		return headers
			.addHeaderIf( options.hasIdempotencyKey(), "Idempotency-Key", options.idempotencyKey() )
			.addHeaderIf( content.hasContent(), "Content-Type", content.contentType() )
			.addHeader( "Accept-Language", options.language() );
	}


	public URL url()
	{
		return resourceUrl;
	}


	public HttpMethod method()
	{
		return method;
	}


	public boolean hasContent()
	{
		return content.hasContent();
	}


	public HttpContent content()
	{
		return content;
	}


	public HttpHeaders headers()
	{
		return headers;
	}


	public HttpParams parameters()
	{
		return parameters;
	}


	public RequestOptions options()
	{
		return options;
	}
}
