/*
 * Copyright (C) 2014-2025 Mambo Solutions Ltd.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package io.mambo.sdk.http;

import static io.mambo.sdk.utils.StringUtils.isNotEmpty;

import java.io.UnsupportedEncodingException;
import java.net.URLEncoder;
import java.nio.charset.StandardCharsets;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.stream.Collectors;

import io.mambo.sdk.exception.MamboException;
import io.mambo.sdk.utils.Multimap;
import io.mambo.sdk.utils.StringUtils;

/**
 * The path and query parameters of the HTTP request
 */
public class HttpParams
{
	private static final Pattern PATH_PARAM_PATTERN = Pattern.compile( "\\{([^}]+)}" );

	private final Multimap<String, String> params;


	private HttpParams( Multimap<String, String> params ) {
		this.params = params;
	}


	public static HttpParams of( Multimap<String, String> params )
	{
		return new HttpParams( params );
	}


	public static HttpParams empty()
	{
		return new HttpParams( new Multimap<>() );
	}


	public String resolvePath( String path )
	{
		if( StringUtils.isBlank( path ) ) {
			return path;
		}

		Matcher matcher = PATH_PARAM_PATTERN.matcher( path );
		String resolvedPath = path;

		while( matcher.find() ) {
			String paramName = matcher.group( 1 );
			String paramValue = params.getFirst( paramName );
			if( isNotEmpty( paramValue ) ) {
				resolvedPath = resolvedPath.replace( "{" + paramName + "}", paramValue );
				params.removeAll( paramName );
			}
		}
		return resolvedPath;
	}


	public String buildQueryString()
	{
		if( params.isEmpty() ) {
			return StringUtils.EMPTY;
		}

		return params.entrySet().stream()
			.flatMap( entry -> entry.getValue().stream()
				.map( value -> encode( entry.getKey() ) + "=" + encode( value ) ) )
			.collect( Collectors.joining( "&", "?", "" ) );
	}


	private String encode( String value )
	{
		try {
			return URLEncoder.encode( value, StandardCharsets.UTF_8.toString() );
		}
		catch( UnsupportedEncodingException exception ) {
			throw new MamboException( "Error encoding URL parameter", exception );
		}
	}
}
