/*
 * Copyright (C) 2014-2025 Mambo Solutions Ltd.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package io.mambo.sdk.http;

import java.nio.charset.StandardCharsets;
import java.util.UUID;

import io.mambo.sdk.service.common.data.FileRequestData;
import io.mambo.sdk.service.common.data.RequestData;
import io.mambo.sdk.utils.JsonUtils;

/**
 * The content of the HTTP request and it's Content-Type header value
 */
public class HttpContent
{
	private final byte[] contentByteArray;
	private final String contentType;


	public HttpContent( byte[] contentByteArray, String contentType ) {
		this.contentByteArray = contentByteArray;
		this.contentType = contentType;
	}


	public static HttpContent empty() {
		return buildEmptyContent();
	}


	public static HttpContent of( RequestData requestData )
	{
		if( requestData == null ) {
			return buildEmptyContent();
		}

		if( requestData instanceof FileRequestData ) {
			return buildMultipartContent( (FileRequestData) requestData );
		}

		return buildJsonContent( requestData );
	}


	private static HttpContent buildEmptyContent()
	{
		return new HttpContent(
			new byte[0],
			"application/json;charset=utf-8" );
	}


	private static HttpContent buildMultipartContent( FileRequestData requestData )
	{
		String boundary = UUID.randomUUID().toString();
		return new HttpContent(
			serializeMultipart( requestData, boundary ),
			String.format( "multipart/form-data;boundary=%s", boundary ) );
	}


	private static byte[] serializeMultipart( FileRequestData requestData, String boundary )
	{
		MultipartBuilder multipart = new MultipartBuilder( boundary );
		multipart.addFileField( "image", requestData.getFile() );
		if( requestData.hasData() ) {
			multipart.addJsonField( "data", requestData.getData() );
		}
		return multipart.build();
	}


	private static HttpContent buildJsonContent( RequestData requestData )
	{
		return new HttpContent(
				serializeJson( requestData ),
				"application/json;charset=utf-8" );
	}


	private static byte[] serializeJson( RequestData requestData )
	{
		return JsonUtils.serialize( requestData ).getBytes( StandardCharsets.UTF_8 );
	}


	public static HttpContent form( String content )
	{
		return new HttpContent(
			content.getBytes( StandardCharsets.UTF_8 ),
			"application/x-www-form-urlencoded;charset=utf-8" );
	}


	public String contentType()
	{
		return contentType;
	}

	public byte[] contentByteArray()
	{
		return contentByteArray;
	}

	public boolean hasContent()
	{
		return contentByteArray.length > 0;
	}
}
