/*
 * Copyright (C) 2014-2025 Mambo Solutions Ltd.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package io.mambo.sdk.exception;

import io.mambo.sdk.exception.oauth.InvalidRequestException;
import io.mambo.sdk.exception.oauth.InvalidTokenException;
import io.mambo.sdk.exception.oauth.OAuthError;
import io.mambo.sdk.exception.oauth.OAuthException;
import io.mambo.sdk.exception.oauth.UnsupportedGrantTypeException;
import io.mambo.sdk.http.HttpStatus;
import io.mambo.sdk.utils.JsonUtils;
import io.mambo.sdk.utils.StringUtils;

public class MamboExceptionFactory
{
	public MamboApiException createMamboApiException(
		String jsonResponse,
		HttpStatus status )
	{
		return createMamboApiException( jsonResponse, status, null );
	}


	public MamboApiException createMamboApiException(
		String jsonResponse,
		HttpStatus status,
		Throwable exception )
	{
		if( isOAuth2Exception( jsonResponse )) {
			return createOAuthException( jsonResponse, status );
		}

		MamboError error = parseError( jsonResponse );

		switch( error.getType() ) {
			case "InvalidIdempotencyKeyException":
			case "IdempotencyKeyAlreadyUsedException":
			case "IdempotencyRequestInProgressException":
				return new IdempotencyException( error, status );
			case "UnauthorizedException":
			case "ForbiddenException":
				return new AuthenticationException( error, status );
			default:
				return new MamboApiException( error, status, exception );
		}
	}


	private boolean isOAuth2Exception( String jsonResponse )
	{
		return StringUtils.contains( jsonResponse, "OAuth2Exception" );
	}


	private OAuthException createOAuthException( String jsonResponse, HttpStatus status )
	{
		OAuthError error = parseOAuthError( jsonResponse );

		switch( error.getError() ) {
			case "invalid_token":
				return new InvalidTokenException( error, status );
			case "invalid_request":
				return new InvalidRequestException( error, status );
			case "unsupported_grant_type":
				return new UnsupportedGrantTypeException( error, status );
			default:
				return new OAuthException( error, status );
		}
	}


	private OAuthError parseOAuthError( String jsonResponse )
	{
		return JsonUtils.deserialize( jsonResponse, OAuthError.class );
	}


	private MamboError parseError( String jsonResponse )
	{
		return JsonUtils.deserialize( jsonResponse, MamboError.class );
	}
}
