/*
 * Copyright (C) 2014-2025 Mambo Solutions Ltd.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package io.mambo.sdk.exception;

import java.util.ArrayList;
import java.util.List;

import com.fasterxml.jackson.annotation.JsonProperty;
import com.fasterxml.jackson.annotation.JsonTypeInfo;
import com.fasterxml.jackson.annotation.JsonTypeInfo.As;
import com.fasterxml.jackson.annotation.JsonTypeInfo.Id;
import com.fasterxml.jackson.annotation.JsonTypeName;

import io.mambo.sdk.utils.ListUtils;
import lombok.EqualsAndHashCode;
import lombok.ToString;


/**
 * Provides details about the error returned by the Mambo API.
 * Each error will always contain a type / message pair. The type
 * is a constant and defines the type of error that occurred. The
 * message attempts to provide more detail on exactly what went wrong.
 */
@ToString
@EqualsAndHashCode
@JsonTypeName( value = "error" )
@JsonTypeInfo( use = Id.NAME, include = As.WRAPPER_OBJECT )
public class MamboError
{
	private List<ParameterError> params = new ArrayList<>();
	private String type;
	private String message;


	/**
	 * The type of Exception being returned. See the API Documentation for
	 * more information on the possible types of Exceptions.
	 *
	 * @return
	 */
	public String getType() { return type; }
	public void setType( String type ) { this.type = type; }

	/**
	 * The error message associated to this exception
	 *
	 * @return
	 */
	public String getMessage() { return message; }
	public void setMessage( String message ) { this.message = message; }

	/**
	 * The detailed error message associated to this exception
	 *
	 * @return
	 */
	public String getDetailedMessage() {
		if( ListUtils.isEmpty( getValidationErrors() ) ) {
			return getMessage();
		}
		StringBuilder builder = new StringBuilder();
		builder.append( message );
		builder.append( " Errors: " );
		builder.append( getValidationErrors() );
		return builder.toString();
	}

	/**
	 * If there are any validation errors these will be returned by this method.
	 * Validation errors can occur when submitting request data to the servers
	 * if the request data is not valid.
	 *
	 * @return
	 */
	@JsonProperty("params")
	public List<ParameterError> getValidationErrors() { return params; }
	public void setValidationErrors( List<ParameterError> params ) { this.params = params; }
}