/*
 * Copyright (C) 2014-2025 Mambo Solutions Ltd.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package io.mambo.sdk.exception;

import java.util.List;

import io.mambo.sdk.http.HttpStatus;
import lombok.ToString;


/**
 * This class represents an error response returned by the Mambo API.
 * Receiving an exception of this type indicates that the request was
 * correctly sent to the API but there was a problem processing the
 * request.
 *
 * Each MamboServiceException is mapped onto a specific error type. The
 * full list of possible errors and HTTP response codes can be found
 * in the documentation.
 */
@ToString
public class MamboApiException extends MamboException
{
	private static final long serialVersionUID = 1L;

	private final transient MamboError error;
	private final HttpStatus status;


	public MamboApiException( MamboError error, HttpStatus status )
	{
		this( error, status, null );
	}


	public MamboApiException( MamboError error, HttpStatus status, Throwable exception )
	{
		super( error.getMessage(), exception );
		this.status = status;
		this.error = error;
	}


	/**
	 * The type of exception that was returned by the Mambo API.
	 * See the API documentation for a comprehensive list.
	 * @return
	 */
	public String getType() {
		return error.getType();
	}


	/**
	 * The message associated with the error returned by the Mambo API.
	 * @return
	 */
	@Override
	public String getMessage() {
		return error.getMessage();
	}


	/**
	 * The validation errors, if any, returned by the Mambo API.
	 * Note: if the error is not validation related, this will return null.
	 * @see ParameterError
	 * @return
	 */
	public List<ParameterError> getValidationErrors() {
		return error.getValidationErrors();
	}


	/**
	 * The HTTP Status
	 * @return
	 */
	public HttpStatus getHttpStatus() {
		return status;
	}
}