/*
 * Copyright (C) 2014-2025 Mambo Solutions Ltd.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package io.mambo.sdk;

import static io.mambo.sdk.utils.Preconditions.checkNotNull;

import java.net.Proxy;

import io.mambo.sdk.http.api.ApiRequestAdapter;
import io.mambo.sdk.http.api.DefaultApiRequestAdapter;
import io.mambo.sdk.http.authenticator.DefaultMamboCredentials;
import io.mambo.sdk.service.activity.ActivitiesService;
import io.mambo.sdk.service.analytics.AnalyticsService;
import io.mambo.sdk.service.behaviour.BehavioursService;
import io.mambo.sdk.service.coupon.CouponsService;
import io.mambo.sdk.service.customfield.CustomFieldsService;
import io.mambo.sdk.service.datastore.DataStoresService;
import io.mambo.sdk.service.kpi.KpisService;
import io.mambo.sdk.service.language.LanguagesService;
import io.mambo.sdk.service.leaderboard.LeaderboardsService;
import io.mambo.sdk.service.notification.NotificationsService;
import io.mambo.sdk.service.point.PointsService;
import io.mambo.sdk.service.reward.RewardsService;
import io.mambo.sdk.service.site.SitesService;
import io.mambo.sdk.service.tag.TagsService;
import io.mambo.sdk.service.user.UsersService;
import io.mambo.sdk.service.webhook.WebhooksService;

public class MamboClient
{
	private final ApiRequestAdapter apiClient;


	public MamboClient( String publicKey, String privateKey ) {
		apiClient = new DefaultApiRequestAdapter(
			new DefaultMamboCredentials( publicKey, privateKey ),
			ClientConfiguration.builder().build() );
	}


	public MamboClient( Builder builder ) {
		apiClient = new DefaultApiRequestAdapter(
			new DefaultMamboCredentials( builder.publicKey, builder.privateKey ),
			builder.configurer.build() );
	}


	public static Builder builder()
	{
		return new Builder();
	}


	public ActivitiesService activities()
	{
		return new ActivitiesService( apiClient );
	}


	public AnalyticsService analytics()
	{
		return new AnalyticsService( apiClient );
	}


	public BehavioursService behaviours()
	{
		return new BehavioursService( apiClient );
	}


	public CouponsService coupons()
	{
		return new CouponsService( apiClient );
	}


	public CustomFieldsService customFields()
	{
		return new CustomFieldsService( apiClient );
	}


	public DataStoresService dataStores()
	{
		return new DataStoresService( apiClient );
	}


	public KpisService kpis()
	{
		return new KpisService( apiClient );
	}


	public LanguagesService languages()
	{
		return new LanguagesService( apiClient );
	}


	public LeaderboardsService leaderboards()
	{
		return new LeaderboardsService( apiClient );
	}


	public NotificationsService notifications()
	{
		return new NotificationsService( apiClient );
	}


	public PointsService points()
	{
		return new PointsService( apiClient );
	}


	public RewardsService rewards()
	{
		return new RewardsService( apiClient );
	}


	public SitesService sites()
	{
		return new SitesService( apiClient );
	}


	public TagsService tags()
	{
		return new TagsService( apiClient );
	}


	public UsersService users()
	{
		return new UsersService( apiClient );
	}


	public WebhooksService webhooks()
	{
		return new WebhooksService( apiClient );
	}


	public static class Builder
	{
		private ClientConfiguration.Builder configurer = ClientConfiguration.builder();
		private String publicKey;
		private String privateKey;


		public Builder credentials( String publicKey, String privateKey )
		{
			this.publicKey = publicKey;
			this.privateKey = privateKey;
			return this;
		}


		public Builder serverBaseUrl( String serverBaseUrl )
		{
			configurer.serverBaseUrl( serverBaseUrl );
			return this;
		}


		public Builder enableLogging( boolean logging )
		{
			configurer.enableLogging( logging );
			return this;
		}


		public Builder sslVerification( boolean sslVerification )
		{
			configurer.sslVerification( sslVerification );
			return this;
		}


		public Builder language( String language )
		{
			configurer.language( language );
			return this;
		}


		public Builder proxyUri( String proxyUri )
		{
			configurer.proxyUri( proxyUri );
			return this;
		}


		public Builder proxy( Proxy proxy )
		{
			configurer.proxy( proxy );
			return this;
		}


		public Builder proxyUsername( String proxyUsername )
		{
			configurer.proxyUsername( proxyUsername );
			return this;
		}


		public Builder proxyPassword( String proxyPassword )
		{
			configurer.proxyPassword( proxyPassword );
			return this;
		}


		public Builder readTimeoutMillis( int readTimeoutMillis )
		{
			configurer.readTimeoutMillis( readTimeoutMillis );
			return this;
		}


		public Builder connectionTimeoutMillis( int connectionTimeoutMillis )
		{
			configurer.connectionTimeoutMillis( connectionTimeoutMillis );
			return this;
		}


		public MamboClient build()
		{
			checkNotNull( publicKey, "publicKey must not be null" );
			checkNotNull( privateKey, "privateKey must not be null" );
			return new MamboClient( this );
		}
	}
}
