/*
 * Copyright (C) 2014-2025 Mambo Solutions Ltd.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package io.mambo.sdk;

import static io.mambo.sdk.utils.Preconditions.checkArgument;
import static io.mambo.sdk.utils.StringUtils.isNotBlank;
import static io.mambo.sdk.utils.StringUtils.startsWith;

import java.net.Proxy;

import io.mambo.sdk.utils.HttpUtils;
import io.mambo.sdk.utils.StringUtils;

/**
 * Properties used to configuration the MamboClient
 */
public class ClientConfiguration
{
	private final String serverBaseUrl;
	private final boolean logging;
	private final boolean sslVerification;
	private final String language;
	private final int readTimeoutMillis;
	private final int connectionTimeoutMillis;
	private final Proxy proxy;
	private final String proxyUsername;
	private final String proxyPassword;


	private ClientConfiguration( Builder builder ) {
		this.serverBaseUrl = builder.serverBaseUrl;
		this.logging = builder.logging;
		this.sslVerification = builder.sslVerification;
		this.language = builder.language;
		this.readTimeoutMillis = builder.readTimeoutMillis;
		this.connectionTimeoutMillis = builder.connectionTimeoutMillis;
		this.proxy = builder.proxy;
		this.proxyUsername = builder.proxyUsername;
		this.proxyPassword = builder.proxyPassword;
	}


	public static Builder builder()
	{
		return new Builder();
	}


	public String getServerBaseUrl()
	{
		return serverBaseUrl;
	}


	public boolean hasLogging()
	{
		return logging;
	}


	public boolean hasSslVerification()
	{
		return sslVerification;
	}


	public String getLanguage()
	{
		return language;
	}


	public int getReadTimeoutMillis()
	{
		return readTimeoutMillis;
	}


	public int getConnectionTimeoutMillis()
	{
		return connectionTimeoutMillis;
	}


	public Proxy getProxy()
	{
		return proxy;
	}


	public String getProxyUsername()
	{
		return proxyUsername;
	}


	public String getProxyPassword()
	{
		return proxyPassword;
	}


	public static class Builder
	{
		private String serverBaseUrl = "https://api.mambo.io";
		private boolean logging = false;
		private boolean sslVerification = true;
		private String language = "en";
		private int connectionTimeoutMillis = 20 * 1000;
		private int readTimeoutMillis = 61 * 1000;
		private Proxy proxy;
		private String proxyUsername;
		private String proxyPassword;


		private Builder() {}


		/**
		 * This is used to set the Server base URL. For example:
		 * https://api.yourdomain.com
		 */
		public Builder serverBaseUrl( String serverBaseUrl )
		{
			checkArgument( isNotBlank( serverBaseUrl ), "serverBaseUrl must not be null or empty" );
			checkArgument( startsWith( serverBaseUrl, "http" ), "serverBaseUrl must include the protocol (e.g. https://)" );
			this.serverBaseUrl = StringUtils.removeEnd( serverBaseUrl, "/" );
			return this;
		}


		/**
		 * Used to enable or disable logging on the MamboClient.
		 */
		public Builder enableLogging( boolean logging )
		{
			this.logging = logging;
			return this;
		}


		/**
		 * Used to enable or disable SSL Verification
		 */
		public Builder sslVerification( boolean sslVerification )
		{
			this.sslVerification = sslVerification;
			return this;
		}


		/**
		 * Set the Language to be used when calling the API
		 */
		public Builder language( String language )
		{
			checkArgument( isNotBlank( language ), "language must not be null or empty" );
			this.language = language;
			return this;
		}


		/**
		 * Proxy URI with port number (e.g. https://192.168.1.1:9400)
		 */
		public Builder proxyUri( String proxyUri )
		{
			checkArgument( isNotBlank( proxyUri ), "proxyUri must not be null or empty" );
			this.proxy = HttpUtils.buildProxy( proxyUri );
			return this;
		}


		/**
		 * Proxy to use for the connection
		 */
		public Builder proxy( Proxy proxy )
		{
			this.proxy = proxy;
			return this;
		}


		/**
		 * The proxy username, leave null if unauthenticated
		 */
		public Builder proxyUsername( String proxyUsername )
		{
			this.proxyUsername = proxyUsername;
			return this;
		}


		/**
		 * The proxy password, leave null if unauthenticated
		 */
		public Builder proxyPassword( String proxyPassword )
		{
			this.proxyPassword = proxyPassword;
			return this;
		}


		/**
		 * Set the timeout to read a response from the server. The default value is 0,
		 * which represents infinity.
		 */
		public Builder readTimeoutMillis( int readTimeoutMillis )
		{
			this.readTimeoutMillis = readTimeoutMillis;
			return this;
		}


		/**
		 * Set the timeout to connect to the server. The default value is 0, which
		 * represents infinity.
		 */
		public Builder connectionTimeoutMillis( int connectionTimeoutMillis )
		{
			this.connectionTimeoutMillis = connectionTimeoutMillis;
			return this;
		}


		public ClientConfiguration build()
		{
			return new ClientConfiguration( this );
		}
	}
}
